<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Visitor extends Model
{
    use HasFactory;

    protected $fillable = [
        'school_id',
        'visitor_name',
        'phone',
        'email',
        'id_proof_type',
        'id_proof_number',
        'purpose',
        'person_to_meet',
        'department',
        'visit_date',
        'check_in_time',
        'check_out_time',
        'photo',
        'remarks',
        'logged_by',
    ];

    protected $casts = [
        'visit_date' => 'date',
        'check_in_time' => 'datetime',
        'check_out_time' => 'datetime',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function loggedBy()
    {
        return $this->belongsTo(User::class, 'logged_by');
    }

    // Scopes
    public function scopeForDate($query, $date)
    {
        return $query->whereDate('visit_date', $date);
    }

    public function scopeCheckedIn($query)
    {
        return $query->whereNotNull('check_in_time')
                     ->whereNull('check_out_time');
    }

    public function scopeCheckedOut($query)
    {
        return $query->whereNotNull('check_out_time');
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('visitor_name', 'like', "%{$search}%")
              ->orWhere('phone', 'like', "%{$search}%")
              ->orWhere('person_to_meet', 'like', "%{$search}%");
        });
    }

    // Accessors
    public function getVisitDurationMinutesAttribute()
    {
        if (!$this->check_in_time || !$this->check_out_time) {
            return null;
        }
        return $this->check_in_time->diffInMinutes($this->check_out_time);
    }
}
