<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Teacher extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'user_id',
        'employee_id',
        'first_name',
        'last_name',
        'email',
        'phone',
        'date_of_birth',
        'gender',
        'blood_group',
        'address',
        'city',
        'state',
        'country',
        'pincode',
        'photo',
        'joining_date',
        'qualification',
        'experience_years',
        'designation',
        'department',
        'specialization',
        'salary',
        'bank_account_number',
        'bank_name',
        'bank_ifsc',
        'emergency_contact_name',
        'emergency_contact_phone',
        'is_active',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'joining_date' => 'date',
        'experience_years' => 'integer',
        'salary' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    protected $hidden = [
        'salary',
        'bank_account_number',
        'bank_name',
        'bank_ifsc',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function subjects()
    {
        return $this->belongsToMany(Subject::class, 'teacher_subject')
                    ->withTimestamps();
    }

    public function classesAsClassTeacher()
    {
        return $this->hasMany(SchoolClass::class, 'class_teacher_id');
    }

    public function sectionsAsSectionTeacher()
    {
        return $this->hasMany(Section::class, 'section_teacher_id');
    }

    public function attendances()
    {
        return $this->hasMany(TeacherAttendance::class);
    }

    public function timetables()
    {
        return $this->hasMany(Timetable::class);
    }

    public function leaveApplications()
    {
        return $this->hasMany(LeaveApplication::class);
    }

    public function classSubjects()
    {
        return $this->hasMany(ClassSubject::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByDepartment($query, $department)
    {
        return $query->where('department', $department);
    }

    public function scopeByDesignation($query, $designation)
    {
        return $query->where('designation', $designation);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('first_name', 'like', "%{$search}%")
              ->orWhere('last_name', 'like', "%{$search}%")
              ->orWhere('employee_id', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%");
        });
    }

    // Accessors
    public function getFullNameAttribute()
    {
        return "{$this->first_name} {$this->last_name}";
    }

    public function getAgeAttribute()
    {
        return $this->date_of_birth?->age;
    }

    public function getFullAddressAttribute()
    {
        return "{$this->address}, {$this->city}, {$this->state}, {$this->country} - {$this->pincode}";
    }

    // Methods
    public function calculateAttendancePercentage($startDate = null, $endDate = null)
    {
        $query = $this->attendances();

        if ($startDate) {
            $query->where('date', '>=', $startDate);
        }

        if ($endDate) {
            $query->where('date', '<=', $endDate);
        }

        $total = $query->count();
        $present = $query->where('status', 'present')->count();

        return $total > 0 ? round(($present / $total) * 100, 2) : 0;
    }
}
