<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class StudentFitnessRecord extends Model
{
    use HasFactory;

    protected $fillable = [
        'student_id',
        'checkup_date',
        'height',
        'weight',
        'bmi',
        'blood_pressure',
        'vision_left',
        'vision_right',
        'dental_status',
        'medical_conditions',
        'allergies',
        'medications',
        'doctor_name',
        'doctor_remarks',
        'next_checkup_date',
    ];

    protected $casts = [
        'checkup_date' => 'date',
        'next_checkup_date' => 'date',
        'height' => 'decimal:2',
        'weight' => 'decimal:2',
        'bmi' => 'decimal:2',
    ];

    // Relationships
    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    // Scopes
    public function scopeForStudent($query, $studentId)
    {
        return $query->where('student_id', $studentId);
    }

    public function scopeLatest($query)
    {
        return $query->orderBy('checkup_date', 'desc');
    }

    public function scopeDueForCheckup($query)
    {
        return $query->where('next_checkup_date', '<=', now());
    }

    // Accessors
    public function getBmiCategoryAttribute()
    {
        if (!$this->bmi) {
            return null;
        }

        if ($this->bmi < 18.5) {
            return 'Underweight';
        } elseif ($this->bmi < 25) {
            return 'Normal';
        } elseif ($this->bmi < 30) {
            return 'Overweight';
        } else {
            return 'Obese';
        }
    }
}
