<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Notifications\Notifiable;
use Tymon\JWTAuth\Contracts\JWTSubject;

class Student extends Authenticatable implements JWTSubject
{
    use HasFactory, SoftDeletes, Notifiable;

    protected $fillable = [
        'school_id',
        'academic_year_id',
        'class_id',
        'section_id',
        'admission_number',
        'roll_number',
        'first_name',
        'last_name',
        'date_of_birth',
        'gender',
        'blood_group',
        'religion',
        'caste',
        'category',
        'mother_tongue',
        'email',
        'password',
        'phone',
        'address',
        'city',
        'state',
        'country',
        'pincode',
        'photo',
        'admission_date',
        'previous_school',
        'medical_history',
        'allergies',
        'emergency_contact_name',
        'emergency_contact_phone',
        'father_name',
        'father_phone',
        'father_occupation',
        'father_email',
        'mother_name',
        'mother_phone',
        'mother_occupation',
        'mother_email',
        'guardian_name',
        'guardian_phone',
        'guardian_relation',
        'is_active',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'admission_date' => 'date',
        'is_active' => 'boolean',
        'password' => 'hashed',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function academicYear()
    {
        return $this->belongsTo(AcademicYear::class);
    }

    public function class()
    {
        return $this->belongsTo(SchoolClass::class, 'class_id');
    }

    public function section()
    {
        return $this->belongsTo(Section::class);
    }

    public function parents()
    {
        return $this->belongsToMany(ParentModel::class, 'parent_student', 'student_id', 'parent_id')
                    ->withPivot('relation')
                    ->withTimestamps();
    }

    public function attendances()
    {
        return $this->hasMany(StudentAttendance::class);
    }

    public function marks()
    {
        return $this->hasMany(Mark::class);
    }

    public function feePayments()
    {
        return $this->hasMany(FeePayment::class);
    }

    public function bookIssues()
    {
        return $this->hasMany(BookIssue::class);
    }

    public function busAssignment()
    {
        return $this->hasOne(BusAssignment::class);
    }

    public function hostelAllocation()
    {
        return $this->hasOne(HostelAllocation::class);
    }

    public function leaveApplications()
    {
        return $this->hasMany(LeaveApplication::class);
    }

    public function eventParticipations()
    {
        return $this->belongsToMany(Event::class, 'event_participants')
                    ->withPivot('participation_type', 'remarks')
                    ->withTimestamps();
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeForClass($query, $classId)
    {
        return $query->where('class_id', $classId);
    }

    public function scopeForSection($query, $sectionId)
    {
        return $query->where('section_id', $sectionId);
    }

    public function scopeByGender($query, $gender)
    {
        return $query->where('gender', $gender);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('first_name', 'like', "%{$search}%")
              ->orWhere('last_name', 'like', "%{$search}%")
              ->orWhere('admission_number', 'like', "%{$search}%")
              ->orWhere('roll_number', 'like', "%{$search}%");
        });
    }

    // Accessors
    public function getFullNameAttribute()
    {
        return "{$this->first_name} {$this->last_name}";
    }

    public function getAgeAttribute()
    {
        return $this->date_of_birth?->age;
    }

    public function getFullAddressAttribute()
    {
        return "{$this->address}, {$this->city}, {$this->state}, {$this->country} - {$this->pincode}";
    }

    // Methods
    public function calculateAttendancePercentage($startDate = null, $endDate = null)
    {
        $query = $this->attendances();

        if ($startDate) {
            $query->where('date', '>=', $startDate);
        }

        if ($endDate) {
            $query->where('date', '<=', $endDate);
        }

        $total = $query->count();
        $present = $query->where('status', 'present')->count();

        return $total > 0 ? round(($present / $total) * 100, 2) : 0;
    }

    // JWT Methods
    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    public function getJWTCustomClaims()
    {
        return [
            'type' => 'student',
            'admission_number' => $this->admission_number,
            'class_id' => $this->class_id,
        ];
    }
}
