<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SportsPlayer extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'student_id',
        'player_name',
        'date_of_birth',
        'gender',
        'blood_group',
        'height',
        'weight',
        'photo',
        'emergency_contact_name',
        'emergency_contact_phone',
        'is_active',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'height' => 'decimal:2',
        'weight' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    public function teams()
    {
        return $this->belongsToMany(SportsTeam::class, 'team_players')
                    ->withPivot('jersey_number', 'position', 'is_captain', 'joined_date')
                    ->withTimestamps();
    }

    public function achievements()
    {
        return $this->hasMany(SportsAchievement::class, 'player_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByGender($query, $gender)
    {
        return $query->where('gender', $gender);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where('player_name', 'like', "%{$search}%");
    }

    // Accessors
    public function getAgeAttribute()
    {
        return $this->date_of_birth?->age;
    }
}
