<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class OnlineExam extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'class_id',
        'section_id',
        'subject_id',
        'exam_id',
        'title',
        'description',
        'instructions',
        'start_datetime',
        'end_datetime',
        'duration_minutes',
        'total_marks',
        'passing_marks',
        'negative_marking',
        'negative_marks',
        'shuffle_questions',
        'show_results_immediately',
        'allow_review',
        'status',
        'is_published',
    ];

    protected $casts = [
        'start_datetime' => 'datetime',
        'end_datetime' => 'datetime',
        'duration_minutes' => 'integer',
        'total_marks' => 'decimal:2',
        'passing_marks' => 'decimal:2',
        'negative_marks' => 'decimal:2',
        'negative_marking' => 'boolean',
        'shuffle_questions' => 'boolean',
        'show_results_immediately' => 'boolean',
        'allow_review' => 'boolean',
        'is_published' => 'boolean',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function class()
    {
        return $this->belongsTo(SchoolClass::class, 'class_id');
    }

    public function section()
    {
        return $this->belongsTo(Section::class);
    }

    public function subject()
    {
        return $this->belongsTo(Subject::class);
    }

    public function exam()
    {
        return $this->belongsTo(Exam::class);
    }

    public function questions()
    {
        return $this->hasMany(OnlineExamQuestion::class, 'online_exam_id');
    }

    public function attempts()
    {
        return $this->belongsToMany(Student::class, 'online_exam_attempts')
                    ->withPivot('start_time', 'end_time', 'marks_obtained', 'status')
                    ->withTimestamps();
    }

    // Scopes
    public function scopePublished($query)
    {
        return $query->where('is_published', true);
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeUpcoming($query)
    {
        return $query->where('start_datetime', '>', now());
    }

    public function scopeOngoing($query)
    {
        return $query->where('start_datetime', '<=', now())
                     ->where('end_datetime', '>=', now());
    }

    public function scopeCompleted($query)
    {
        return $query->where('end_datetime', '<', now());
    }

    public function scopeForClass($query, $classId)
    {
        return $query->where('class_id', $classId);
    }

    // Accessors
    public function getIsActiveAttribute()
    {
        $now = now();
        return $this->start_datetime <= $now && $this->end_datetime >= $now;
    }

    public function getQuestionCountAttribute()
    {
        return $this->questions()->count();
    }
}
