<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class LmsSubmission extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'assignment_id',
        'student_id',
        'submission_date',
        'submission_text',
        'attachment_url',
        'marks_obtained',
        'feedback',
        'graded_by',
        'graded_at',
        'status',
        'is_late',
    ];

    protected $casts = [
        'submission_date' => 'datetime',
        'marks_obtained' => 'decimal:2',
        'graded_at' => 'datetime',
        'is_late' => 'boolean',
    ];

    // Relationships
    public function assignment()
    {
        return $this->belongsTo(LmsAssignment::class, 'assignment_id');
    }

    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    public function gradedBy()
    {
        return $this->belongsTo(Teacher::class, 'graded_by');
    }

    // Scopes
    public function scopeForAssignment($query, $assignmentId)
    {
        return $query->where('assignment_id', $assignmentId);
    }

    public function scopeForStudent($query, $studentId)
    {
        return $query->where('student_id', $studentId);
    }

    public function scopeGraded($query)
    {
        return $query->where('status', 'graded');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeLateSubmissions($query)
    {
        return $query->where('is_late', true);
    }

    // Accessors
    public function getIsGradedAttribute()
    {
        return $this->status === 'graded';
    }

    public function getPercentageAttribute()
    {
        if (!$this->marks_obtained || !$this->assignment) {
            return 0;
        }
        return round(($this->marks_obtained / $this->assignment->max_marks) * 100, 2);
    }
}
