<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class LibraryBook extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'book_category_id',
        'title',
        'isbn',
        'author',
        'publisher',
        'edition',
        'publication_year',
        'language',
        'pages',
        'description',
        'cover_image',
        'total_copies',
        'available_copies',
        'price',
        'rack_number',
        'is_active',
    ];

    protected $casts = [
        'publication_year' => 'integer',
        'pages' => 'integer',
        'total_copies' => 'integer',
        'available_copies' => 'integer',
        'price' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function bookCategory()
    {
        return $this->belongsTo(BookCategory::class, 'book_category_id');
    }

    public function copies()
    {
        return $this->hasMany(LibraryBookCopy::class, 'library_book_id');
    }

    public function reservations()
    {
        return $this->hasMany(BookReservation::class, 'library_book_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeAvailable($query)
    {
        return $query->where('available_copies', '>', 0);
    }

    public function scopeForCategory($query, $categoryId)
    {
        return $query->where('book_category_id', $categoryId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('title', 'like', "%{$search}%")
              ->orWhere('author', 'like', "%{$search}%")
              ->orWhere('isbn', 'like', "%{$search}%")
              ->orWhere('publisher', 'like', "%{$search}%");
        });
    }

    // Accessors
    public function getIsAvailableAttribute()
    {
        return $this->available_copies > 0;
    }

    public function getIssuedCopiesAttribute()
    {
        return $this->total_copies - $this->available_copies;
    }
}
