<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class InventoryItem extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'item_category_id',
        'item_name',
        'item_code',
        'description',
        'unit',
        'minimum_stock_level',
        'maximum_stock_level',
        'reorder_level',
        'unit_price',
        'image',
        'is_active',
    ];

    protected $casts = [
        'minimum_stock_level' => 'integer',
        'maximum_stock_level' => 'integer',
        'reorder_level' => 'integer',
        'unit_price' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function itemCategory()
    {
        return $this->belongsTo(ItemCategory::class, 'item_category_id');
    }

    public function stocks()
    {
        return $this->hasMany(ItemStock::class, 'item_id');
    }

    public function stockAdditions()
    {
        return $this->hasMany(StockAddition::class, 'item_id');
    }

    public function stockIssues()
    {
        return $this->hasMany(StockIssue::class, 'item_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeForCategory($query, $categoryId)
    {
        return $query->where('item_category_id', $categoryId);
    }

    public function scopeLowStock($query)
    {
        return $query->whereHas('stocks', function($q) {
            $q->whereRaw('available_quantity <= reorder_level');
        });
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('item_name', 'like', "%{$search}%")
              ->orWhere('item_code', 'like', "%{$search}%");
        });
    }

    // Accessors
    public function getTotalStockAttribute()
    {
        return $this->stocks()->sum('available_quantity');
    }

    public function getIsLowStockAttribute()
    {
        return $this->getTotalStockAttribute() <= $this->reorder_level;
    }
}
