<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Event extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'title',
        'description',
        'event_type',
        'event_date',
        'start_time',
        'end_time',
        'venue',
        'organizer',
        'max_participants',
        'target_audience',
        'class_id',
        'is_mandatory',
        'is_active',
    ];

    protected $casts = [
        'event_date' => 'date',
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'max_participants' => 'integer',
        'is_mandatory' => 'boolean',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function class()
    {
        return $this->belongsTo(SchoolClass::class, 'class_id');
    }

    public function participants()
    {
        return $this->belongsToMany(Student::class, 'event_participants')
                    ->withPivot('participation_type', 'remarks')
                    ->withTimestamps();
    }

    public function eventParticipants()
    {
        return $this->hasMany(EventParticipant::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeUpcoming($query)
    {
        return $query->where('event_date', '>=', now()->toDateString());
    }

    public function scopeCompleted($query)
    {
        return $query->where('event_date', '<', now()->toDateString());
    }

    public function scopeByType($query, $type)
    {
        return $query->where('event_type', $type);
    }

    public function scopeMandatory($query)
    {
        return $query->where('is_mandatory', true);
    }

    // Accessors
    public function getAvailableSlotsAttribute()
    {
        return $this->max_participants - $this->participants()->count();
    }

    public function getIsFullAttribute()
    {
        return $this->participants()->count() >= $this->max_participants;
    }
}
