<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Driver extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'staff_id',
        'driver_name',
        'email',
        'phone',
        'date_of_birth',
        'address',
        'city',
        'state',
        'pincode',
        'photo',
        'license_number',
        'license_type',
        'license_issue_date',
        'license_expiry_date',
        'blood_group',
        'emergency_contact_name',
        'emergency_contact_phone',
        'joining_date',
        'is_active',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'license_issue_date' => 'date',
        'license_expiry_date' => 'date',
        'joining_date' => 'date',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function staff()
    {
        return $this->belongsTo(Staff::class);
    }

    public function vehicles()
    {
        return $this->belongsToMany(Vehicle::class, 'vehicle_driver_assignments')
                    ->withPivot('assigned_date', 'is_primary')
                    ->withTimestamps();
    }

    public function routes()
    {
        return $this->hasMany(TransportRoute::class, 'driver_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeExpiringLicense($query, $days = 30)
    {
        return $query->whereBetween('license_expiry_date', [now(), now()->addDays($days)]);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('driver_name', 'like', "%{$search}%")
              ->orWhere('phone', 'like', "%{$search}%")
              ->orWhere('license_number', 'like', "%{$search}%");
        });
    }

    // Accessors
    public function getIsLicenseValidAttribute()
    {
        return $this->license_expiry_date && $this->license_expiry_date >= now();
    }

    public function getAgeAttribute()
    {
        return $this->date_of_birth?->age;
    }
}
