<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Bus extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'bus_number',
        'registration_number',
        'model',
        'capacity',
        'driver_name',
        'driver_phone',
        'driver_license',
        'conductor_name',
        'conductor_phone',
        'insurance_expiry',
        'fitness_expiry',
        'is_active',
    ];

    protected $casts = [
        'capacity' => 'integer',
        'insurance_expiry' => 'date',
        'fitness_expiry' => 'date',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function routes()
    {
        return $this->hasMany(BusRoute::class);
    }

    public function assignments()
    {
        return $this->hasMany(BusAssignment::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeInsuranceExpiring($query, $days = 30)
    {
        return $query->whereBetween('insurance_expiry', [now(), now()->addDays($days)]);
    }

    public function scopeFitnessExpiring($query, $days = 30)
    {
        return $query->whereBetween('fitness_expiry', [now(), now()->addDays($days)]);
    }

    // Accessors
    public function getAvailableSeatsAttribute()
    {
        return $this->capacity - $this->assignments()->count();
    }

    public function getIsFullAttribute()
    {
        return $this->assignments()->count() >= $this->capacity;
    }
}
