<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Budget extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'account_head_id',
        'financial_year',
        'budget_amount',
        'allocated_amount',
        'spent_amount',
        'start_date',
        'end_date',
        'status',
        'remarks',
    ];

    protected $casts = [
        'budget_amount' => 'decimal:2',
        'allocated_amount' => 'decimal:2',
        'spent_amount' => 'decimal:2',
        'start_date' => 'date',
        'end_date' => 'date',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function accountHead()
    {
        return $this->belongsTo(AccountHead::class, 'account_head_id');
    }

    // Scopes
    public function scopeForAccountHead($query, $accountHeadId)
    {
        return $query->where('account_head_id', $accountHeadId);
    }

    public function scopeForFinancialYear($query, $year)
    {
        return $query->where('financial_year', $year);
    }

    public function scopeActive($query)
    {
        return $query->where('start_date', '<=', now())
                     ->where('end_date', '>=', now());
    }

    // Accessors
    public function getRemainingAmountAttribute()
    {
        return $this->budget_amount - $this->spent_amount;
    }

    public function getUtilizationPercentageAttribute()
    {
        if ($this->budget_amount == 0) {
            return 0;
        }
        return round(($this->spent_amount / $this->budget_amount) * 100, 2);
    }
}
