<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class BookIssue extends Model
{
    use HasFactory;

    protected $fillable = [
        'book_id',
        'student_id',
        'issued_date',
        'due_date',
        'return_date',
        'fine_amount',
        'remarks',
        'issued_by',
        'returned_to',
        'status',
    ];

    protected $casts = [
        'issued_date' => 'date',
        'due_date' => 'date',
        'return_date' => 'date',
        'fine_amount' => 'decimal:2',
    ];

    // Relationships
    public function book()
    {
        return $this->belongsTo(Book::class);
    }

    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    public function issuedBy()
    {
        return $this->belongsTo(User::class, 'issued_by');
    }

    public function returnedTo()
    {
        return $this->belongsTo(User::class, 'returned_to');
    }

    // Scopes
    public function scopeIssued($query)
    {
        return $query->whereNull('return_date');
    }

    public function scopeReturned($query)
    {
        return $query->whereNotNull('return_date');
    }

    public function scopeOverdue($query)
    {
        return $query->whereNull('return_date')
                     ->where('due_date', '<', now()->toDateString());
    }

    public function scopeForStudent($query, $studentId)
    {
        return $query->where('student_id', $studentId);
    }

    public function scopeForBook($query, $bookId)
    {
        return $query->where('book_id', $bookId);
    }

    // Accessors
    public function getIsOverdueAttribute()
    {
        return !$this->return_date && $this->due_date < now();
    }

    public function getDaysOverdueAttribute()
    {
        if (!$this->is_overdue) {
            return 0;
        }
        return now()->diffInDays($this->due_date);
    }

    // Methods
    public function calculateFine($finePerDay = 5)
    {
        if ($this->is_overdue) {
            return $this->days_overdue * $finePerDay;
        }
        return 0;
    }

    public function returnBook($returnedTo = null)
    {
        $this->return_date = now();
        $this->returned_to = $returnedTo;
        $this->fine_amount = $this->calculateFine();
        $this->status = 'returned';
        $this->save();

        $this->book->incrementAvailableQuantity();
    }
}
