<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Book extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'school_id',
        'book_category_id',
        'title',
        'isbn',
        'author',
        'publisher',
        'publication_year',
        'edition',
        'pages',
        'price',
        'quantity',
        'available_quantity',
        'rack_number',
        'description',
        'cover_image',
        'is_active',
    ];

    protected $casts = [
        'publication_year' => 'integer',
        'pages' => 'integer',
        'price' => 'decimal:2',
        'quantity' => 'integer',
        'available_quantity' => 'integer',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function category()
    {
        return $this->belongsTo(BookCategory::class, 'book_category_id');
    }

    public function issues()
    {
        return $this->hasMany(BookIssue::class);
    }

    public function currentIssues()
    {
        return $this->hasMany(BookIssue::class)->whereNull('return_date');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeAvailable($query)
    {
        return $query->where('available_quantity', '>', 0);
    }

    public function scopeByCategory($query, $categoryId)
    {
        return $query->where('book_category_id', $categoryId);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('title', 'like', "%{$search}%")
              ->orWhere('isbn', 'like', "%{$search}%")
              ->orWhere('author', 'like', "%{$search}%");
        });
    }

    // Accessors
    public function getIsAvailableAttribute()
    {
        return $this->available_quantity > 0;
    }

    public function getIssuedQuantityAttribute()
    {
        return $this->quantity - $this->available_quantity;
    }

    // Methods
    public function decrementAvailableQuantity()
    {
        if ($this->available_quantity > 0) {
            $this->decrement('available_quantity');
            return true;
        }
        return false;
    }

    public function incrementAvailableQuantity()
    {
        if ($this->available_quantity < $this->quantity) {
            $this->increment('available_quantity');
            return true;
        }
        return false;
    }
}
