<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Alumni extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'alumni';

    protected $fillable = [
        'school_id',
        'student_id',
        'first_name',
        'last_name',
        'email',
        'phone',
        'date_of_birth',
        'gender',
        'photo',
        'batch_year',
        'passed_out_class',
        'current_occupation',
        'company_name',
        'designation',
        'address',
        'city',
        'state',
        'country',
        'linkedin_profile',
        'facebook_profile',
        'achievements',
        'is_active',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'batch_year' => 'integer',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function school()
    {
        return $this->belongsTo(School::class);
    }

    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    public function events()
    {
        return $this->belongsToMany(AlumniEvent::class, 'alumni_event_registrations')
                    ->withPivot('registration_date', 'payment_status', 'attendance_status')
                    ->withTimestamps();
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByBatchYear($query, $year)
    {
        return $query->where('batch_year', $year);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('first_name', 'like', "%{$search}%")
              ->orWhere('last_name', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%")
              ->orWhere('phone', 'like', "%{$search}%");
        });
    }

    // Accessors
    public function getFullNameAttribute()
    {
        return "{$this->first_name} {$this->last_name}";
    }
}
