<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Api\ApiController;
use App\Models\Timetable;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class TimetableController extends ApiController
{
    /**
     * Display a listing of timetables with pagination and search
     */
    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 15);
        $search = $request->get('search');
        $sortBy = $request->get('sort_by', 'day_of_week');
        $sortOrder = $request->get('sort_order', 'asc');

        $query = Timetable::with([
            'class',
            'section',
            'subject',
            'teacher',
            'period',
            'academicYear'
        ]);

        // Apply search
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->whereHas('teacher', function($tq) use ($search) {
                    $tq->where('first_name', 'like', "%{$search}%")
                       ->orWhere('last_name', 'like', "%{$search}%");
                })
                ->orWhereHas('subject', function($sq) use ($search) {
                    $sq->where('name', 'like', "%{$search}%");
                })
                ->orWhereHas('class', function($cq) use ($search) {
                    $cq->where('name', 'like', "%{$search}%");
                });
            });
        }

        // Apply filters
        if ($request->has('class_id') && $request->class_id) {
            $query->where('class_id', $request->class_id);
        }

        if ($request->has('section_id') && $request->section_id) {
            $query->where('section_id', $request->section_id);
        }

        if ($request->has('teacher_id') && $request->teacher_id) {
            $query->where('teacher_id', $request->teacher_id);
        }

        if ($request->has('subject_id') && $request->subject_id) {
            $query->where('subject_id', $request->subject_id);
        }

        if ($request->has('day_of_week') && $request->day_of_week) {
            $query->where('day_of_week', $request->day_of_week);
        }

        if ($request->has('academic_year_id') && $request->academic_year_id) {
            $query->where('academic_year_id', $request->academic_year_id);
        }

        if ($request->has('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        // Apply sorting
        $query = $this->applySorting($query, $sortBy, $sortOrder);

        // Secondary sort by period
        if ($sortBy !== 'period_id') {
            $query->orderBy('period_id', 'asc');
        }

        return $this->paginatedResponse($query, $perPage);
    }

    /**
     * Store a newly created timetable entry
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'academic_year_id' => 'required|exists:academic_years,id',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'subject_id' => 'required|exists:subjects,id',
            'teacher_id' => 'required|exists:teachers,id',
            'period_id' => 'required|exists:periods,id',
            'day_of_week' => 'required|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'room_number' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        // Check for conflicts
        $conflict = $this->checkTimetableConflict(
            $request->class_id,
            $request->section_id,
            $request->day_of_week,
            $request->period_id,
            $request->teacher_id
        );

        if ($conflict) {
            return $this->errorResponse($conflict, 400);
        }

        DB::beginTransaction();
        try {
            $timetable = Timetable::create($request->all());

            DB::commit();

            return $this->successResponse(
                $timetable->load(['class', 'section', 'subject', 'teacher', 'period']),
                'Timetable entry created successfully',
                201
            );
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to create timetable: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Display the specified timetable entry
     */
    public function show($id)
    {
        $timetable = Timetable::with([
            'class',
            'section',
            'subject',
            'teacher',
            'period',
            'academicYear'
        ])->find($id);

        if (!$timetable) {
            return $this->errorResponse('Timetable entry not found', 404);
        }

        return $this->successResponse($timetable);
    }

    /**
     * Update the specified timetable entry
     */
    public function update(Request $request, $id)
    {
        $timetable = Timetable::find($id);

        if (!$timetable) {
            return $this->errorResponse('Timetable entry not found', 404);
        }

        $validator = Validator::make($request->all(), [
            'academic_year_id' => 'sometimes|exists:academic_years,id',
            'class_id' => 'sometimes|exists:school_classes,id',
            'section_id' => 'sometimes|exists:sections,id',
            'subject_id' => 'sometimes|exists:subjects,id',
            'teacher_id' => 'sometimes|exists:teachers,id',
            'period_id' => 'sometimes|exists:periods,id',
            'day_of_week' => 'sometimes|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'room_number' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        // Check for conflicts if relevant fields are being updated
        if ($request->hasAny(['class_id', 'section_id', 'day_of_week', 'period_id', 'teacher_id'])) {
            $conflict = $this->checkTimetableConflict(
                $request->class_id ?? $timetable->class_id,
                $request->section_id ?? $timetable->section_id,
                $request->day_of_week ?? $timetable->day_of_week,
                $request->period_id ?? $timetable->period_id,
                $request->teacher_id ?? $timetable->teacher_id,
                $id
            );

            if ($conflict) {
                return $this->errorResponse($conflict, 400);
            }
        }

        DB::beginTransaction();
        try {
            $timetable->update($request->all());

            DB::commit();

            return $this->successResponse(
                $timetable->load(['class', 'section', 'subject', 'teacher', 'period']),
                'Timetable entry updated successfully'
            );
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to update timetable: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Remove the specified timetable entry
     */
    public function destroy($id)
    {
        $timetable = Timetable::find($id);

        if (!$timetable) {
            return $this->errorResponse('Timetable entry not found', 404);
        }

        DB::beginTransaction();
        try {
            $timetable->delete();
            DB::commit();

            return $this->successResponse(null, 'Timetable entry deleted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to delete timetable: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get timetable by class and section
     */
    public function getByClass(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'nullable|exists:sections,id',
            'day_of_week' => 'nullable|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $query = Timetable::with(['subject', 'teacher', 'period'])
            ->where('class_id', $request->class_id)
            ->where('is_active', true);

        if ($request->has('section_id') && $request->section_id) {
            $query->where('section_id', $request->section_id);
        }

        if ($request->has('day_of_week') && $request->day_of_week) {
            $query->where('day_of_week', $request->day_of_week);
        }

        $timetable = $query->orderBy('day_of_week')
            ->orderBy('period_id')
            ->get();

        // Group by day of week
        $groupedTimetable = $timetable->groupBy('day_of_week');

        return $this->successResponse([
            'timetable' => $timetable,
            'grouped_by_day' => $groupedTimetable,
        ]);
    }

    /**
     * Get timetable for a specific teacher
     */
    public function getByTeacher(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'teacher_id' => 'required|exists:teachers,id',
            'day_of_week' => 'nullable|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $query = Timetable::with(['class', 'section', 'subject', 'period'])
            ->where('teacher_id', $request->teacher_id)
            ->where('is_active', true);

        if ($request->has('day_of_week') && $request->day_of_week) {
            $query->where('day_of_week', $request->day_of_week);
        }

        $timetable = $query->orderBy('day_of_week')
            ->orderBy('period_id')
            ->get();

        // Group by day of week
        $groupedTimetable = $timetable->groupBy('day_of_week');

        return $this->successResponse([
            'timetable' => $timetable,
            'grouped_by_day' => $groupedTimetable,
        ]);
    }

    /**
     * Bulk create timetable entries
     */
    public function bulkCreate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'entries' => 'required|array',
            'entries.*.academic_year_id' => 'required|exists:academic_years,id',
            'entries.*.class_id' => 'required|exists:school_classes,id',
            'entries.*.section_id' => 'required|exists:sections,id',
            'entries.*.subject_id' => 'required|exists:subjects,id',
            'entries.*.teacher_id' => 'required|exists:teachers,id',
            'entries.*.period_id' => 'required|exists:periods,id',
            'entries.*.day_of_week' => 'required|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'entries.*.room_number' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        DB::beginTransaction();
        try {
            $created = [];
            $errors = [];

            foreach ($request->entries as $index => $entry) {
                // Check for conflicts
                $conflict = $this->checkTimetableConflict(
                    $entry['class_id'],
                    $entry['section_id'],
                    $entry['day_of_week'],
                    $entry['period_id'],
                    $entry['teacher_id']
                );

                if ($conflict) {
                    $errors[] = [
                        'index' => $index,
                        'entry' => $entry,
                        'error' => $conflict,
                    ];
                    continue;
                }

                $entry['is_active'] = true;
                $created[] = Timetable::create($entry);
            }

            DB::commit();

            return $this->successResponse([
                'created' => $created,
                'errors' => $errors,
                'total_created' => count($created),
                'total_errors' => count($errors),
            ], 'Bulk timetable creation completed', 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to create timetables: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Clone timetable from one class/section to another
     */
    public function clone(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'source_class_id' => 'required|exists:school_classes,id',
            'source_section_id' => 'required|exists:sections,id',
            'target_class_id' => 'required|exists:school_classes,id',
            'target_section_id' => 'required|exists:sections,id',
            'academic_year_id' => 'required|exists:academic_years,id',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        DB::beginTransaction();
        try {
            $sourceTimetables = Timetable::where('class_id', $request->source_class_id)
                ->where('section_id', $request->source_section_id)
                ->where('is_active', true)
                ->get();

            if ($sourceTimetables->isEmpty()) {
                return $this->errorResponse('No timetable found for source class/section', 404);
            }

            $cloned = [];
            foreach ($sourceTimetables as $timetable) {
                $cloned[] = Timetable::create([
                    'academic_year_id' => $request->academic_year_id,
                    'class_id' => $request->target_class_id,
                    'section_id' => $request->target_section_id,
                    'subject_id' => $timetable->subject_id,
                    'teacher_id' => $timetable->teacher_id,
                    'period_id' => $timetable->period_id,
                    'day_of_week' => $timetable->day_of_week,
                    'room_number' => $timetable->room_number,
                    'is_active' => true,
                ]);
            }

            DB::commit();

            return $this->successResponse([
                'cloned_entries' => $cloned,
                'total_cloned' => count($cloned),
            ], 'Timetable cloned successfully', 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to clone timetable: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get weekly timetable view
     */
    public function weeklyView(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'nullable|exists:sections,id',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $query = Timetable::with(['subject', 'teacher', 'period'])
            ->where('class_id', $request->class_id)
            ->where('is_active', true);

        if ($request->has('section_id') && $request->section_id) {
            $query->where('section_id', $request->section_id);
        }

        $timetables = $query->orderBy('day_of_week')
            ->orderBy('period_id')
            ->get();

        // Create weekly matrix
        $weekDays = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
        $weeklyMatrix = [];

        foreach ($weekDays as $day) {
            $dayTimetables = $timetables->where('day_of_week', $day)->sortBy('period_id');
            $weeklyMatrix[$day] = $dayTimetables->map(function($tt) {
                return [
                    'period_id' => $tt->period_id,
                    'period' => $tt->period,
                    'subject' => $tt->subject->name,
                    'teacher' => $tt->teacher->first_name . ' ' . $tt->teacher->last_name,
                    'room_number' => $tt->room_number,
                ];
            })->values();
        }

        return $this->successResponse([
            'weekly_timetable' => $weeklyMatrix,
            'days' => $weekDays,
        ]);
    }

    /**
     * Delete all timetable entries for a class/section
     */
    public function deleteByClass(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        DB::beginTransaction();
        try {
            $deleted = Timetable::where('class_id', $request->class_id)
                ->where('section_id', $request->section_id)
                ->delete();

            DB::commit();

            return $this->successResponse([
                'deleted_count' => $deleted,
            ], 'Timetable entries deleted successfully');

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to delete timetables: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Check for timetable conflicts
     */
    private function checkTimetableConflict($classId, $sectionId, $dayOfWeek, $periodId, $teacherId, $excludeId = null)
    {
        // Check if slot is already occupied for this class/section
        $classConflict = Timetable::where('class_id', $classId)
            ->where('section_id', $sectionId)
            ->where('day_of_week', $dayOfWeek)
            ->where('period_id', $periodId)
            ->where('is_active', true)
            ->when($excludeId, function($q) use ($excludeId) {
                return $q->where('id', '!=', $excludeId);
            })
            ->first();

        if ($classConflict) {
            return 'This time slot is already occupied for this class/section';
        }

        // Check if teacher is already assigned to another class at this time
        $teacherConflict = Timetable::where('teacher_id', $teacherId)
            ->where('day_of_week', $dayOfWeek)
            ->where('period_id', $periodId)
            ->where('is_active', true)
            ->when($excludeId, function($q) use ($excludeId) {
                return $q->where('id', '!=', $excludeId);
            })
            ->first();

        if ($teacherConflict) {
            return 'Teacher is already assigned to another class at this time';
        }

        return null;
    }

    /**
     * Get conflict check
     */
    public function checkConflict(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'day_of_week' => 'required|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'period_id' => 'required|exists:periods,id',
            'teacher_id' => 'required|exists:teachers,id',
            'exclude_id' => 'nullable|exists:timetables,id',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $conflict = $this->checkTimetableConflict(
            $request->class_id,
            $request->section_id,
            $request->day_of_week,
            $request->period_id,
            $request->teacher_id,
            $request->exclude_id
        );

        return $this->successResponse([
            'has_conflict' => $conflict !== null,
            'conflict_message' => $conflict,
        ]);
    }
}
