<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Api\ApiController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class RoleController extends ApiController
{
    /**
     * Display a listing of roles with pagination and search
     */
    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 15);
        $search = $request->get('search');
        $sortBy = $request->get('sort_by', 'name');
        $sortOrder = $request->get('sort_order', 'asc');

        // TODO: Implement when roles table exists
        // For now, return sample roles
        $roles = [
            [
                'id' => 1,
                'name' => 'Super Admin',
                'slug' => 'super-admin',
                'description' => 'Full system access',
                'is_system' => true,
                'permissions_count' => 100,
                'users_count' => 2,
                'created_at' => now(),
            ],
            [
                'id' => 2,
                'name' => 'Principal',
                'slug' => 'principal',
                'description' => 'School principal access',
                'is_system' => true,
                'permissions_count' => 85,
                'users_count' => 1,
                'created_at' => now(),
            ],
            [
                'id' => 3,
                'name' => 'Teacher',
                'slug' => 'teacher',
                'description' => 'Teacher access for classes',
                'is_system' => true,
                'permissions_count' => 45,
                'users_count' => 25,
                'created_at' => now(),
            ],
            [
                'id' => 4,
                'name' => 'Accountant',
                'slug' => 'accountant',
                'description' => 'Fee and finance management',
                'is_system' => false,
                'permissions_count' => 20,
                'users_count' => 3,
                'created_at' => now(),
            ],
            [
                'id' => 5,
                'name' => 'Librarian',
                'slug' => 'librarian',
                'description' => 'Library management',
                'is_system' => false,
                'permissions_count' => 15,
                'users_count' => 2,
                'created_at' => now(),
            ],
        ];

        if ($search) {
            $roles = array_filter($roles, function($role) use ($search) {
                return stripos($role['name'], $search) !== false ||
                       stripos($role['description'], $search) !== false;
            });
        }

        return $this->successResponse([
            'data' => array_values($roles),
            'total' => count($roles),
            'message' => 'Roles will be fetched from database once roles table is created',
        ]);
    }

    /**
     * Store a newly created role
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:roles,name',
            'slug' => 'required|string|max:255|unique:roles,slug',
            'description' => 'nullable|string',
            'permission_ids' => 'nullable|array',
            'permission_ids.*' => 'exists:permissions,id',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        // TODO: Implement when roles table exists
        $roleData = [
            'id' => rand(100, 999),
            'name' => $request->name,
            'slug' => $request->slug,
            'description' => $request->description,
            'is_system' => false,
            'created_at' => now(),
        ];

        return $this->successResponse(
            $roleData,
            'Role will be created in database once roles table is created',
            201
        );
    }

    /**
     * Display the specified role
     */
    public function show($id)
    {
        // TODO: Fetch from database
        $role = [
            'id' => $id,
            'name' => 'Sample Role',
            'slug' => 'sample-role',
            'description' => 'Sample role description',
            'permissions' => [],
            'users_count' => 0,
        ];

        return $this->successResponse($role);
    }

    /**
     * Update the specified role
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255|unique:roles,name,' . $id,
            'slug' => 'sometimes|string|max:255|unique:roles,slug,' . $id,
            'description' => 'nullable|string',
            'permission_ids' => 'nullable|array',
            'permission_ids.*' => 'exists:permissions,id',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        // TODO: Update in database
        return $this->successResponse(
            ['id' => $id, 'updated' => true],
            'Role will be updated in database once roles table is created'
        );
    }

    /**
     * Remove the specified role
     */
    public function destroy($id)
    {
        // TODO: Check if role is system role
        // TODO: Check if role has users
        // TODO: Delete from database

        return $this->successResponse(
            null,
            'Role will be deleted from database once roles table is created'
        );
    }

    /**
     * Assign permissions to role
     */
    public function assignPermissions(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'permission_ids' => 'required|array',
            'permission_ids.*' => 'exists:permissions,id',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        // TODO: Sync permissions in database
        // $role->permissions()->sync($request->permission_ids);

        return $this->successResponse(
            ['role_id' => $id, 'permissions_assigned' => count($request->permission_ids)],
            'Permissions assigned successfully'
        );
    }

    /**
     * Clone a role
     */
    public function clone(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:roles,name',
            'slug' => 'required|string|max:255|unique:roles,slug',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        // TODO: Clone role with permissions
        return $this->successResponse(
            ['cloned_role_id' => rand(100, 999)],
            'Role cloned successfully',
            201
        );
    }
}
