<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Api\ApiController;
use App\Models\Notice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class NoticeController extends ApiController
{
    /**
     * Display a listing of notices with pagination and search
     */
    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 15);
        $search = $request->get('search');
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');

        $query = Notice::with(['school', 'createdBy']);

        // Apply search
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Apply filters
        if ($request->has('school_id') && $request->school_id) {
            $query->where('school_id', $request->school_id);
        }

        if ($request->has('target_audience') && $request->target_audience) {
            $query->where('target_audience', $request->target_audience);
        }

        if ($request->has('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        // Apply sorting
        $query = $this->applySorting($query, $sortBy, $sortOrder);

        return $this->paginatedResponse($query, $perPage);
    }

    /**
     * Store a newly created notice
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'school_id' => 'required|exists:schools,id',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'target_audience' => 'required|in:all,students,teachers,parents,staff',
            'publish_date' => 'required|date',
            'expiry_date' => 'nullable|date|after:publish_date',
            'attachment' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $data = $request->all();
        $data['created_by'] = auth('admin')->id();

        $notice = Notice::create($data);

        return $this->successResponse(
            $notice->load(['school', 'createdBy']),
            'Notice created successfully',
            201
        );
    }

    /**
     * Display the specified notice
     */
    public function show($id)
    {
        $notice = Notice::with(['school', 'createdBy'])->find($id);

        if (!$notice) {
            return $this->errorResponse('Notice not found', 404);
        }

        return $this->successResponse($notice);
    }

    /**
     * Update the specified notice
     */
    public function update(Request $request, $id)
    {
        $notice = Notice::find($id);

        if (!$notice) {
            return $this->errorResponse('Notice not found', 404);
        }

        $validator = Validator::make($request->all(), [
            'school_id' => 'sometimes|exists:schools,id',
            'title' => 'sometimes|string|max:255',
            'description' => 'sometimes|string',
            'target_audience' => 'sometimes|in:all,students,teachers,parents,staff',
            'publish_date' => 'sometimes|date',
            'expiry_date' => 'nullable|date|after:publish_date',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $notice->update($request->all());

        return $this->successResponse(
            $notice->load(['school', 'createdBy']),
            'Notice updated successfully'
        );
    }

    /**
     * Remove the specified notice
     */
    public function destroy($id)
    {
        $notice = Notice::find($id);

        if (!$notice) {
            return $this->errorResponse('Notice not found', 404);
        }

        $notice->delete();

        return $this->successResponse(null, 'Notice deleted successfully');
    }
}
