<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Api\ApiController;
use App\Models\Hostel;
use App\Models\HostelAllocation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class HostelController extends ApiController
{
    /**
     * Display a listing of hostels with pagination and search
     */
    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 15);
        $search = $request->get('search');
        $sortBy = $request->get('sort_by', 'name');
        $sortOrder = $request->get('sort_order', 'asc');

        $query = Hostel::with(['school', 'rooms']);

        // Apply search
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('type', 'like', "%{$search}%")
                  ->orWhere('address', 'like', "%{$search}%");
            });
        }

        // Apply filters
        if ($request->has('school_id') && $request->school_id) {
            $query->where('school_id', $request->school_id);
        }

        if ($request->has('type') && $request->type) {
            $query->where('type', $request->type);
        }

        if ($request->has('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        // Apply sorting
        $query = $this->applySorting($query, $sortBy, $sortOrder);

        return $this->paginatedResponse($query, $perPage);
    }

    /**
     * Store a newly created hostel
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'school_id' => 'required|exists:schools,id',
            'name' => 'required|string|max:255',
            'type' => 'required|in:boys,girls,mixed',
            'address' => 'nullable|string',
            'warden_name' => 'nullable|string',
            'warden_phone' => 'nullable|string',
            'capacity' => 'required|integer|min:1',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $hostel = Hostel::create($request->all());

        return $this->successResponse(
            $hostel->load('school'),
            'Hostel created successfully',
            201
        );
    }

    /**
     * Display the specified hostel
     */
    public function show($id)
    {
        $hostel = Hostel::with(['school', 'rooms', 'allocations'])->find($id);

        if (!$hostel) {
            return $this->errorResponse('Hostel not found', 404);
        }

        return $this->successResponse($hostel);
    }

    /**
     * Update the specified hostel
     */
    public function update(Request $request, $id)
    {
        $hostel = Hostel::find($id);

        if (!$hostel) {
            return $this->errorResponse('Hostel not found', 404);
        }

        $validator = Validator::make($request->all(), [
            'school_id' => 'sometimes|exists:schools,id',
            'name' => 'sometimes|string|max:255',
            'type' => 'sometimes|in:boys,girls,mixed',
            'capacity' => 'sometimes|integer|min:1',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $hostel->update($request->all());

        return $this->successResponse(
            $hostel->load('school'),
            'Hostel updated successfully'
        );
    }

    /**
     * Remove the specified hostel
     */
    public function destroy($id)
    {
        $hostel = Hostel::find($id);

        if (!$hostel) {
            return $this->errorResponse('Hostel not found', 404);
        }

        $hostel->delete();

        return $this->successResponse(null, 'Hostel deleted successfully');
    }

    /**
     * Allocate hostel to student
     */
    public function allocate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'hostel_id' => 'required|exists:hostels,id',
            'room_id' => 'required|exists:hostel_rooms,id',
            'student_id' => 'required|exists:students,id',
            'allocation_date' => 'required|date',
            'fee_amount' => 'nullable|numeric|min:0',
            'remarks' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $allocation = HostelAllocation::create($request->all());

        return $this->successResponse(
            $allocation->load(['hostel', 'room', 'student']),
            'Hostel allocated successfully',
            201
        );
    }
}
