<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Api\ApiController;
use App\Models\FeePayment;
use App\Models\Student;
use App\Models\FeeStructure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class FeeReportController extends ApiController
{
    /**
     * Get comprehensive fee reports with filters
     */
    public function index(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'report_type' => 'required|in:summary,detailed,class_wise,defaulters,collection',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'class_id' => 'nullable|exists:school_classes,id',
            'section_id' => 'nullable|exists:sections,id',
            'school_id' => 'nullable|exists:schools,id',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $reportType = $request->report_type;

        switch ($reportType) {
            case 'summary':
                return $this->summaryReport($request);
            case 'detailed':
                return $this->detailedReport($request);
            case 'class_wise':
                return $this->classWiseReport($request);
            case 'defaulters':
                return $this->defaultersReport($request);
            case 'collection':
                return $this->collectionReport($request);
            default:
                return $this->errorResponse('Invalid report type', 400);
        }
    }

    /**
     * Fee summary report
     */
    private function summaryReport(Request $request)
    {
        $query = FeePayment::whereBetween('payment_date', [$request->start_date, $request->end_date]);

        if ($request->has('school_id') && $request->school_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('school_id', $request->school_id);
            });
        }

        if ($request->has('class_id') && $request->class_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('class_id', $request->class_id);
            });
        }

        $summary = [
            'total_transactions' => $query->count(),
            'total_amount_due' => (clone $query)->sum('amount_due'),
            'total_amount_paid' => (clone $query)->sum('amount_paid'),
            'total_balance' => (clone $query)->sum('balance_amount'),
            'total_discount' => (clone $query)->sum('discount_amount'),
            'total_fine' => (clone $query)->sum('fine_amount'),
            'collection_percentage' => 0,
        ];

        if ($summary['total_amount_due'] > 0) {
            $summary['collection_percentage'] = round(
                ($summary['total_amount_paid'] / $summary['total_amount_due']) * 100,
                2
            );
        }

        // Payment method breakdown
        $paymentMethods = (clone $query)
            ->select('payment_method', DB::raw('COUNT(*) as count'), DB::raw('SUM(amount_paid) as total'))
            ->groupBy('payment_method')
            ->get();

        // Payment status breakdown
        $paymentStatus = (clone $query)
            ->select('payment_status', DB::raw('COUNT(*) as count'), DB::raw('SUM(amount_paid) as total'))
            ->groupBy('payment_status')
            ->get();

        return $this->successResponse([
            'summary' => $summary,
            'payment_methods' => $paymentMethods,
            'payment_status' => $paymentStatus,
            'period' => [
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
            ],
        ]);
    }

    /**
     * Detailed fee report with all transactions
     */
    private function detailedReport(Request $request)
    {
        $perPage = $request->get('per_page', 50);

        $query = FeePayment::with([
            'student.class',
            'student.section',
            'feeStructure.feeType',
            'collectedBy'
        ])->whereBetween('payment_date', [$request->start_date, $request->end_date]);

        if ($request->has('school_id') && $request->school_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('school_id', $request->school_id);
            });
        }

        if ($request->has('class_id') && $request->class_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('class_id', $request->class_id);
            });
        }

        if ($request->has('section_id') && $request->section_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('section_id', $request->section_id);
            });
        }

        $query->orderBy('payment_date', 'desc');

        return $this->paginatedResponse($query, $perPage);
    }

    /**
     * Class-wise fee collection report
     */
    private function classWiseReport(Request $request)
    {
        $query = FeePayment::with(['student.class'])
            ->whereBetween('payment_date', [$request->start_date, $request->end_date]);

        if ($request->has('school_id') && $request->school_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('school_id', $request->school_id);
            });
        }

        $classWiseData = $query->get()->groupBy(function($item) {
            return $item->student->class_id;
        })->map(function($classPayments, $classId) {
            $className = $classPayments->first()->student->class->name ?? 'Unknown';

            return [
                'class_id' => $classId,
                'class_name' => $className,
                'total_students' => $classPayments->pluck('student_id')->unique()->count(),
                'total_transactions' => $classPayments->count(),
                'total_amount_due' => $classPayments->sum('amount_due'),
                'total_amount_paid' => $classPayments->sum('amount_paid'),
                'total_balance' => $classPayments->sum('balance_amount'),
                'total_discount' => $classPayments->sum('discount_amount'),
                'collection_percentage' => $classPayments->sum('amount_due') > 0
                    ? round(($classPayments->sum('amount_paid') / $classPayments->sum('amount_due')) * 100, 2)
                    : 0,
            ];
        })->values();

        return $this->successResponse([
            'class_wise_data' => $classWiseData,
            'period' => [
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
            ],
        ]);
    }

    /**
     * Fee defaulters report (students with pending payments)
     */
    private function defaultersReport(Request $request)
    {
        $perPage = $request->get('per_page', 50);

        $query = FeePayment::with([
            'student.class',
            'student.section',
            'feeStructure.feeType'
        ])
        ->where('payment_status', '!=', 'paid')
        ->where('balance_amount', '>', 0);

        if ($request->has('start_date') && $request->start_date) {
            $query->where('payment_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->where('payment_date', '<=', $request->end_date);
        }

        if ($request->has('school_id') && $request->school_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('school_id', $request->school_id);
            });
        }

        if ($request->has('class_id') && $request->class_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('class_id', $request->class_id);
            });
        }

        $query->orderBy('balance_amount', 'desc');

        $data = $query->paginate($perPage);

        $totalDefaulters = $data->total();
        $totalOutstanding = FeePayment::where('payment_status', '!=', 'paid')
            ->where('balance_amount', '>', 0)
            ->sum('balance_amount');

        return response()->json([
            'success' => true,
            'data' => $data->items(),
            'summary' => [
                'total_defaulters' => $totalDefaulters,
                'total_outstanding_amount' => $totalOutstanding,
            ],
            'pagination' => [
                'total' => $data->total(),
                'per_page' => $data->perPage(),
                'current_page' => $data->currentPage(),
                'last_page' => $data->lastPage(),
            ],
        ]);
    }

    /**
     * Daily collection report
     */
    private function collectionReport(Request $request)
    {
        $query = FeePayment::whereBetween('payment_date', [$request->start_date, $request->end_date]);

        if ($request->has('school_id') && $request->school_id) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('school_id', $request->school_id);
            });
        }

        // Daily collection breakdown
        $dailyCollection = (clone $query)
            ->select(
                DB::raw('DATE(payment_date) as date'),
                DB::raw('COUNT(*) as transaction_count'),
                DB::raw('SUM(amount_paid) as total_collected')
            )
            ->groupBy('date')
            ->orderBy('date', 'asc')
            ->get();

        // Collector-wise breakdown
        $collectorWise = (clone $query)
            ->select(
                'collected_by',
                DB::raw('COUNT(*) as transaction_count'),
                DB::raw('SUM(amount_paid) as total_collected')
            )
            ->with('collectedBy:id,name')
            ->groupBy('collected_by')
            ->get();

        $totalCollection = $query->sum('amount_paid');
        $totalTransactions = $query->count();

        return $this->successResponse([
            'daily_collection' => $dailyCollection,
            'collector_wise' => $collectorWise,
            'summary' => [
                'total_collection' => $totalCollection,
                'total_transactions' => $totalTransactions,
                'average_per_transaction' => $totalTransactions > 0
                    ? round($totalCollection / $totalTransactions, 2)
                    : 0,
            ],
            'period' => [
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
            ],
        ]);
    }

    /**
     * Export report (CSV/PDF) - Future implementation
     */
    public function export(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'report_type' => 'required|in:summary,detailed,class_wise,defaulters,collection',
            'format' => 'required|in:csv,pdf,excel',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        // TODO: Implement export functionality
        return $this->successResponse([
            'message' => 'Export functionality will be implemented',
            'format' => $request->format,
        ]);
    }
}
