<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Api\ApiController;
use App\Models\Exam;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ExamController extends ApiController
{
    /**
     * Display a listing of exams with pagination and search
     */
    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 15);
        $search = $request->get('search');
        $sortBy = $request->get('sort_by', 'start_date');
        $sortOrder = $request->get('sort_order', 'desc');

        $query = Exam::with(['school', 'academicYear']);

        // Apply search
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('exam_type', 'like', "%{$search}%");
            });
        }

        // Apply filters
        if ($request->has('school_id') && $request->school_id) {
            $query->where('school_id', $request->school_id);
        }

        if ($request->has('academic_year_id') && $request->academic_year_id) {
            $query->where('academic_year_id', $request->academic_year_id);
        }

        if ($request->has('exam_type') && $request->exam_type) {
            $query->where('exam_type', $request->exam_type);
        }

        // Apply sorting
        $query = $this->applySorting($query, $sortBy, $sortOrder);

        return $this->paginatedResponse($query, $perPage);
    }

    /**
     * Store a newly created exam
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'school_id' => 'required|exists:schools,id',
            'academic_year_id' => 'required|exists:academic_years,id',
            'name' => 'required|string|max:255',
            'exam_type' => 'required|in:midterm,final,quarterly,unit_test,other',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'result_publish_date' => 'nullable|date',
            'description' => 'nullable|string',
            'is_published' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $exam = Exam::create($request->all());

        return $this->successResponse(
            $exam->load(['school', 'academicYear']),
            'Exam created successfully',
            201
        );
    }

    /**
     * Display the specified exam
     */
    public function show($id)
    {
        $exam = Exam::with(['school', 'academicYear', 'schedules', 'marks'])->find($id);

        if (!$exam) {
            return $this->errorResponse('Exam not found', 404);
        }

        return $this->successResponse($exam);
    }

    /**
     * Update the specified exam
     */
    public function update(Request $request, $id)
    {
        $exam = Exam::find($id);

        if (!$exam) {
            return $this->errorResponse('Exam not found', 404);
        }

        $validator = Validator::make($request->all(), [
            'school_id' => 'sometimes|exists:schools,id',
            'academic_year_id' => 'sometimes|exists:academic_years,id',
            'name' => 'sometimes|string|max:255',
            'exam_type' => 'sometimes|in:midterm,final,quarterly,unit_test,other',
            'start_date' => 'sometimes|date',
            'end_date' => 'sometimes|date|after_or_equal:start_date',
            'result_publish_date' => 'nullable|date',
            'description' => 'nullable|string',
            'is_published' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $exam->update($request->all());

        return $this->successResponse(
            $exam->load(['school', 'academicYear']),
            'Exam updated successfully'
        );
    }

    /**
     * Remove the specified exam
     */
    public function destroy($id)
    {
        $exam = Exam::find($id);

        if (!$exam) {
            return $this->errorResponse('Exam not found', 404);
        }

        $exam->delete();

        return $this->successResponse(null, 'Exam deleted successfully');
    }

    /**
     * Publish exam results
     */
    public function publishResults($id)
    {
        $exam = Exam::find($id);

        if (!$exam) {
            return $this->errorResponse('Exam not found', 404);
        }

        $exam->update([
            'is_published' => true,
            'result_publish_date' => now(),
        ]);

        return $this->successResponse($exam, 'Exam results published successfully');
    }
}
