<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Api\ApiController;
use App\Models\Student;
use App\Models\Teacher;
use App\Models\Staff;
use App\Models\StudentAttendance;
use App\Models\FeePayment;
use App\Models\Notice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends ApiController
{
    /**
     * Get dashboard statistics
     */
    public function index(Request $request)
    {
        $schoolId = $request->get('school_id');
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        // Build queries with school filter
        $studentsQuery = Student::query();
        $teachersQuery = Teacher::query();
        $staffQuery = Staff::query();
        $attendanceQuery = StudentAttendance::query();
        $feePaymentQuery = FeePayment::query();
        $noticeQuery = Notice::query();

        if ($schoolId) {
            $studentsQuery->where('school_id', $schoolId);
            $teachersQuery->where('school_id', $schoolId);
            $staffQuery->where('school_id', $schoolId);
            $noticeQuery->where('school_id', $schoolId);
        }

        // Student statistics
        $totalStudents = $studentsQuery->count();
        $activeStudents = (clone $studentsQuery)->where('is_active', true)->count();
        $maleStudents = (clone $studentsQuery)->where('gender', 'male')->count();
        $femaleStudents = (clone $studentsQuery)->where('gender', 'female')->count();

        // Teacher & Staff statistics
        $totalTeachers = $teachersQuery->count();
        $activeTeachers = (clone $teachersQuery)->where('is_active', true)->count();
        $totalStaff = $staffQuery->count();

        // Attendance statistics (current month)
        $attendanceQuery->whereBetween('date', [$startDate, $endDate]);
        $totalAttendance = $attendanceQuery->count();
        $presentCount = (clone $attendanceQuery)->where('status', 'present')->count();
        $absentCount = (clone $attendanceQuery)->where('status', 'absent')->count();
        $attendancePercentage = $totalAttendance > 0
            ? round(($presentCount / $totalAttendance) * 100, 2)
            : 0;

        // Fee statistics (current month)
        $feePaymentQuery->whereBetween('payment_date', [$startDate, $endDate]);
        $totalFeeCollected = (clone $feePaymentQuery)->sum('amount_paid');
        $totalFeeTransactions = $feePaymentQuery->count();

        // Recent notices
        $recentNotices = $noticeQuery->where('is_active', true)
            ->where('publish_date', '<=', now())
            ->where(function($q) {
                $q->whereNull('expiry_date')
                  ->orWhere('expiry_date', '>=', now());
            })
            ->latest()
            ->limit(5)
            ->get();

        // Upcoming exams
        $upcomingExams = DB::table('exams')
            ->where('start_date', '>=', now())
            ->orderBy('start_date', 'asc')
            ->limit(5)
            ->get();

        // Class-wise student distribution
        $classDistribution = Student::select('class_id', DB::raw('count(*) as count'))
            ->with('class:id,name')
            ->groupBy('class_id')
            ->get();

        return $this->successResponse([
            'students' => [
                'total' => $totalStudents,
                'active' => $activeStudents,
                'male' => $maleStudents,
                'female' => $femaleStudents,
            ],
            'teachers' => [
                'total' => $totalTeachers,
                'active' => $activeTeachers,
            ],
            'staff' => [
                'total' => $totalStaff,
            ],
            'attendance' => [
                'total' => $totalAttendance,
                'present' => $presentCount,
                'absent' => $absentCount,
                'percentage' => $attendancePercentage,
                'period' => [
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                ],
            ],
            'fees' => [
                'total_collected' => $totalFeeCollected,
                'total_transactions' => $totalFeeTransactions,
                'period' => [
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                ],
            ],
            'recent_notices' => $recentNotices,
            'upcoming_exams' => $upcomingExams,
            'class_distribution' => $classDistribution,
        ]);
    }
}
