<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Api\ApiController;
use App\Models\Book;
use App\Models\BookIssue;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class BookController extends ApiController
{
    /**
     * Display a listing of books with pagination and search
     */
    public function index(Request $request)
    {
        $perPage = $request->get('per_page', 15);
        $search = $request->get('search');
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');

        $query = Book::with(['category', 'school']);

        // Apply search
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('isbn', 'like', "%{$search}%")
                  ->orWhere('author', 'like', "%{$search}%")
                  ->orWhere('publisher', 'like', "%{$search}%");
            });
        }

        // Apply filters
        if ($request->has('category_id') && $request->category_id) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->has('availability_status') && $request->availability_status) {
            $query->where('availability_status', $request->availability_status);
        }

        // Apply sorting
        $query = $this->applySorting($query, $sortBy, $sortOrder);

        return $this->paginatedResponse($query, $perPage);
    }

    /**
     * Store a newly created book
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'school_id' => 'required|exists:schools,id',
            'category_id' => 'required|exists:book_categories,id',
            'title' => 'required|string|max:255',
            'isbn' => 'required|string|unique:books,isbn',
            'author' => 'required|string',
            'publisher' => 'nullable|string',
            'publication_year' => 'nullable|integer',
            'edition' => 'nullable|string',
            'language' => 'nullable|string',
            'pages' => 'nullable|integer',
            'price' => 'nullable|numeric',
            'rack_number' => 'nullable|string',
            'quantity' => 'required|integer|min:0',
            'available_quantity' => 'required|integer|min:0',
            'availability_status' => 'nullable|in:available,issued,lost,damaged',
            'description' => 'nullable|string',
            'cover_image' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $book = Book::create($request->all());

        return $this->successResponse(
            $book->load(['category', 'school']),
            'Book created successfully',
            201
        );
    }

    /**
     * Display the specified book
     */
    public function show($id)
    {
        $book = Book::with(['category', 'school', 'issues'])->find($id);

        if (!$book) {
            return $this->errorResponse('Book not found', 404);
        }

        return $this->successResponse($book);
    }

    /**
     * Update the specified book
     */
    public function update(Request $request, $id)
    {
        $book = Book::find($id);

        if (!$book) {
            return $this->errorResponse('Book not found', 404);
        }

        $validator = Validator::make($request->all(), [
            'school_id' => 'sometimes|exists:schools,id',
            'category_id' => 'sometimes|exists:book_categories,id',
            'title' => 'sometimes|string|max:255',
            'isbn' => 'sometimes|string|unique:books,isbn,' . $id,
            'author' => 'sometimes|string',
            'publisher' => 'nullable|string',
            'publication_year' => 'nullable|integer',
            'quantity' => 'sometimes|integer|min:0',
            'available_quantity' => 'sometimes|integer|min:0',
            'availability_status' => 'nullable|in:available,issued,lost,damaged',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        $book->update($request->all());

        return $this->successResponse(
            $book->load(['category', 'school']),
            'Book updated successfully'
        );
    }

    /**
     * Remove the specified book
     */
    public function destroy($id)
    {
        $book = Book::find($id);

        if (!$book) {
            return $this->errorResponse('Book not found', 404);
        }

        $book->delete();

        return $this->successResponse(null, 'Book deleted successfully');
    }

    /**
     * Issue a book to a student
     */
    public function issue(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'book_id' => 'required|exists:books,id',
            'student_id' => 'required|exists:students,id',
            'issue_date' => 'required|date',
            'due_date' => 'required|date|after:issue_date',
            'remarks' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        DB::beginTransaction();
        try {
            $book = Book::find($request->book_id);

            if ($book->available_quantity <= 0) {
                return $this->errorResponse('Book is not available for issue', 400);
            }

            $bookIssue = BookIssue::create([
                'book_id' => $request->book_id,
                'student_id' => $request->student_id,
                'issue_date' => $request->issue_date,
                'due_date' => $request->due_date,
                'status' => 'issued',
                'remarks' => $request->remarks,
            ]);

            $book->decrement('available_quantity');
            $book->update(['availability_status' => $book->available_quantity > 0 ? 'available' : 'issued']);

            DB::commit();

            return $this->successResponse(
                $bookIssue->load(['book', 'student']),
                'Book issued successfully',
                201
            );
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to issue book: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Return a book
     */
    public function returnBook(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'book_issue_id' => 'required|exists:book_issues,id',
            'return_date' => 'required|date',
            'condition' => 'nullable|in:good,damaged,lost',
            'fine_amount' => 'nullable|numeric|min:0',
            'remarks' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse('Validation failed', 422, $validator->errors());
        }

        DB::beginTransaction();
        try {
            $bookIssue = BookIssue::with('book')->find($request->book_issue_id);

            if ($bookIssue->status === 'returned') {
                return $this->errorResponse('Book already returned', 400);
            }

            $bookIssue->update([
                'return_date' => $request->return_date,
                'status' => 'returned',
                'condition' => $request->condition ?? 'good',
                'fine_amount' => $request->fine_amount ?? 0,
                'return_remarks' => $request->remarks,
            ]);

            $book = $bookIssue->book;
            $book->increment('available_quantity');
            $book->update(['availability_status' => 'available']);

            DB::commit();

            return $this->successResponse(
                $bookIssue->load(['book', 'student']),
                'Book returned successfully'
            );
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to return book: ' . $e->getMessage(), 500);
        }
    }
}
