# API Routes Organization

## Overview
The School ERP API routes are organized into separate files by user type and functionality for better maintainability and scalability.

## Route File Structure

```
routes/
├── api.php          # Main API routes file (aggregator)
├── admin.php        # Admin, Teachers, Staff, Principal routes
├── student.php      # Student routes
├── parent.php       # Parent/Guardian routes
└── public.php       # Public CMS routes (no authentication)
```

---

## Route Prefixes

All routes are prefixed with `/api/` by default in Laravel.

| User Type | Prefix | Authentication | Example |
|-----------|--------|----------------|---------|
| Admin | `/api/admin` | JWT (admin guard) | `/api/admin/profile` |
| Student | `/api/student` | JWT (student guard) | `/api/student/attendance` |
| Parent | `/api/parent` | JWT (parent guard) | `/api/parent/children` |
| Public | `/api/public` | None | `/api/public/admissions` |

---

## 1. Main API Routes (`routes/api.php`)

### Purpose
- Entry point for all API routes
- Includes separate route files
- Provides API health check endpoint
- Defines fallback route for 404 errors

### Endpoints

#### Health Check
```
GET /api/
```
Returns API information, version, and available endpoints.

**Response:**
```json
{
  "success": true,
  "message": "School ERP API - JWT Authentication",
  "version": "1.0.0",
  "endpoints": {
    "admin": "/api/admin",
    "student": "/api/student",
    "parent": "/api/parent",
    "public": "/api/public"
  }
}
```

#### Fallback (404)
```
ANY /api/*
```
Returns 404 for undefined routes.

---

## 2. Admin Routes (`routes/admin.php`)

### Prefix: `/api/admin`
### Authentication: JWT (admin guard)
### User Types: Admin, Teachers, Staff, Principal

### Public Routes (No Auth)
```
POST /api/admin/register          # Register new admin/staff
POST /api/admin/login             # Login
```

### Protected Routes (Auth Required)

#### Profile Management
```
GET  /api/admin/profile           # Get profile
PUT  /api/admin/profile           # Update profile
POST /api/admin/change-password   # Change password
```

#### Token Management
```
POST /api/admin/refresh           # Refresh JWT token
POST /api/admin/logout            # Logout
```

#### Student Management (Commented - Ready to implement)
```
GET    /api/admin/students              # List all students
POST   /api/admin/students              # Create student
GET    /api/admin/students/{id}         # Get student details
PUT    /api/admin/students/{id}         # Update student
DELETE /api/admin/students/{id}         # Delete student
GET    /api/admin/students/{id}/attendance  # Student attendance
GET    /api/admin/students/{id}/marks   # Student marks
```

#### More Admin Features (See file for complete list)
- Teacher Management
- Class/Section/Subject Management
- Exam Management
- Fee Management
- Attendance Management
- Timetable Management
- Library Management
- Transport Management
- Hostel Management
- Reports
- Communication (Notices, SMS, Email)
- System Settings
- User & Role Management

---

## 3. Student Routes (`routes/student.php`)

### Prefix: `/api/student`
### Authentication: JWT (student guard)
### User Types: Students

### Public Routes (No Auth)
```
POST /api/student/register        # Register new student
POST /api/student/login           # Login
```

### Protected Routes (Auth Required)

#### Profile Management
```
GET  /api/student/profile         # Get profile
PUT  /api/student/profile         # Update profile
POST /api/student/change-password # Change password
```

#### Token Management
```
POST /api/student/refresh         # Refresh JWT token
POST /api/student/logout          # Logout
```

#### Academic Features (Commented - Ready to implement)
```
GET /api/student/dashboard        # Student dashboard
GET /api/student/class-info       # Class information
GET /api/student/subjects         # Enrolled subjects
GET /api/student/timetable        # Class timetable
```

#### Attendance
```
GET /api/student/attendance       # Attendance records
GET /api/student/attendance/summary  # Attendance summary
GET /api/student/attendance/monthly  # Monthly attendance
```

#### Exams & Results
```
GET /api/student/exams            # Upcoming exams
GET /api/student/results          # Exam results
GET /api/student/marksheet/{exam_id}  # Download marksheet
```

#### More Student Features (See file for complete list)
- Homework & Assignments
- Fee Management
- Library (Issued books, Search, Reserve)
- Transport Information
- Notices & Announcements
- Leave Applications
- Events & Calendar
- Documents
- Online Classes (LMS)
- Online Exams
- Sports & Activities

---

## 4. Parent Routes (`routes/parent.php`)

### Prefix: `/api/parent`
### Authentication: JWT (parent guard)
### User Types: Parents/Guardians

### Public Routes (No Auth)
```
POST /api/parent/register         # Register new parent
POST /api/parent/login            # Login
```

### Protected Routes (Auth Required)

#### Profile Management
```
GET  /api/parent/profile          # Get profile
PUT  /api/parent/profile          # Update profile
POST /api/parent/change-password  # Change password
```

#### Token Management
```
POST /api/parent/refresh          # Refresh JWT token
POST /api/parent/logout           # Logout
```

#### Children Management (Commented - Ready to implement)
```
GET /api/parent/children          # List all children
GET /api/parent/children/{id}     # Child details
GET /api/parent/children/{id}/profile  # Child profile
```

#### Children's Academic Performance
```
GET /api/parent/children/{id}/attendance      # Attendance
GET /api/parent/children/{id}/subjects        # Subjects
GET /api/parent/children/{id}/timetable       # Timetable
GET /api/parent/children/{id}/exams           # Exams
GET /api/parent/children/{id}/results         # Results
GET /api/parent/children/{id}/homework        # Homework
```

#### Fee Management
```
GET  /api/parent/children/{id}/fees           # Fee details
GET  /api/parent/children/{id}/fees/pending   # Pending fees
POST /api/parent/children/{id}/fees/pay       # Pay fees
GET  /api/parent/children/{id}/fees/receipt/{id}  # Fee receipt
```

#### More Parent Features (See file for complete list)
- Communication with School
- Teacher Communication
- Parent-Teacher Meetings
- Feedback
- Leave Applications
- Transport Tracking
- Library Information
- Notices & Events
- Health & Fitness Records
- Progress Reports
- Complaints & Grievances

---

## 5. Public CMS Routes (`routes/public.php`)

### Prefix: `/api/public`
### Authentication: None (Public access)
### Purpose: Website content, admissions, news, events

### Public Information (Commented - Ready to implement)

#### School Information
```
GET /api/public/school-info       # School basic info
GET /api/public/about-us          # About school
GET /api/public/contact           # Contact information
POST /api/public/contact          # Submit contact form
```

#### Admissions
```
GET  /api/public/admissions       # Admission information
GET  /api/public/admission-process  # Process details
POST /api/public/admission-enquiry  # Submit enquiry
POST /api/public/admission-application  # Apply for admission
GET  /api/public/fee-structure    # Fee structure
```

#### News & Events
```
GET /api/public/news              # Latest news
GET /api/public/news/{id}         # News details
GET /api/public/events            # Upcoming events
GET /api/public/events/{id}       # Event details
GET /api/public/announcements     # Public announcements
```

#### More Public Features (See file for complete list)
- Faculty & Staff Information
- Gallery (Photos, Videos, Albums)
- Achievements (Academic, Sports, Awards)
- Alumni Section
- Downloads (Prospectus, Forms, Syllabus)
- Blog
- Testimonials
- Career/Jobs
- Newsletter Subscription
- FAQ
- Policies & Terms

---

## How Routes Are Loaded

### In `routes/api.php`:
```php
// Admin routes
Route::prefix('admin')->group(function () {
    require __DIR__.'/admin.php';
});

// Student routes
Route::prefix('student')->group(function () {
    require __DIR__.'/student.php';
});

// Parent routes
Route::prefix('parent')->group(function () {
    require __DIR__.'/parent.php';
});

// Public routes
Route::prefix('public')->group(function () {
    require __DIR__.'/public.php';
});
```

---

## Route Testing

### View All Routes
```bash
php artisan route:list
```

### View API Routes Only
```bash
php artisan route:list --path=api
```

### View Admin Routes
```bash
php artisan route:list --path=api/admin
```

### View Student Routes
```bash
php artisan route:list --path=api/student
```

### Clear Route Cache
```bash
php artisan route:clear
```

---

## Adding New Routes

### For Admin Features
Edit `routes/admin.php` and uncomment or add routes:
```php
Route::middleware('auth:admin')->group(function () {
    Route::apiResource('students', StudentController::class);
});
```

### For Student Features
Edit `routes/student.php`:
```php
Route::middleware('auth:student')->group(function () {
    Route::get('/homework', [StudentHomeworkController::class, 'index']);
});
```

### For Parent Features
Edit `routes/parent.php`:
```php
Route::middleware('auth:parent')->group(function () {
    Route::get('/children', [ParentChildrenController::class, 'index']);
});
```

### For Public CMS
Edit `routes/public.php` (no authentication):
```php
Route::get('/news', [PublicNewsController::class, 'index']);
```

---

## Authentication Guards

### Admin Guard
```php
Route::middleware('auth:admin')->group(function () {
    // Admin routes
});
```

### Student Guard
```php
Route::middleware('auth:student')->group(function () {
    // Student routes
});
```

### Parent Guard
```php
Route::middleware('auth:parent')->group(function () {
    // Parent routes
});
```

### Multiple Guards (Shared Routes)
```php
Route::middleware('auth:admin,student,parent')->group(function () {
    // Shared routes accessible by all
});
```

---

## Best Practices

### 1. Keep Routes Organized
- Each user type has its own route file
- Group related routes together
- Comment unused routes instead of deleting

### 2. Use Resource Controllers
```php
Route::apiResource('students', StudentController::class);
```
This automatically creates:
- GET /students (index)
- POST /students (store)
- GET /students/{id} (show)
- PUT /students/{id} (update)
- DELETE /students/{id} (destroy)

### 3. Route Naming
```php
Route::get('/profile', [AuthController::class, 'profile'])->name('admin.profile');
```

### 4. Route Grouping
```php
Route::prefix('reports')->group(function () {
    Route::get('/students', [ReportController::class, 'students']);
    Route::get('/attendance', [ReportController::class, 'attendance']);
});
```

### 5. Middleware Application
```php
// Apply to single route
Route::get('/profile', [AuthController::class, 'profile'])->middleware('auth:admin');

// Apply to group
Route::middleware('auth:admin')->group(function () {
    // Routes here
});
```

---

## Route Statistics

### Current Routes (Active)
- **Admin**: 8 routes (7 authenticated + 1 public)
- **Student**: 8 routes (7 authenticated + 1 public)
- **Parent**: 8 routes (7 authenticated + 1 public)
- **Public**: 0 routes (ready to uncomment)
- **Total**: 24+ routes

### Planned Routes (Commented)
- **Admin**: 50+ additional routes
- **Student**: 40+ additional routes
- **Parent**: 45+ additional routes
- **Public**: 60+ CMS routes
- **Total**: 195+ additional routes ready to implement

---

## Next Steps

1. **Create Controllers**: Implement controllers for uncommented routes
2. **Add Validation**: Create Form Request classes for validation
3. **Implement Services**: Business logic in service classes
4. **Add Tests**: Unit and feature tests for all routes
5. **API Documentation**: Generate API docs using tools like Swagger/OpenAPI

---

## Quick Reference Commands

```bash
# View routes
php artisan route:list

# Clear route cache
php artisan route:clear

# Cache routes (production)
php artisan route:cache

# Test route
curl http://localhost:8000/api/

# Test with authentication
curl -H "Authorization: Bearer {token}" http://localhost:8000/api/admin/profile
```

---

**Version:** 1.0.0
**Last Updated:** January 2026
**Maintainer:** Development Team
