# Multi-Authentication API Setup Guide

## Overview
This School ERP API supports **three separate authentication systems**:
1. **Students** - Student portal access
2. **Parents** - Parent portal access
3. **Admin/Users** - Admin, Teachers, Staff, Principal access

Each has independent authentication with Laravel Sanctum tokens.

---

## Architecture

### Models
- **Student** → `students` table
- **ParentModel** → `parents` table
- **User** → `users` table

All models extend `Authenticatable` and use:
- `HasApiTokens` (Sanctum)
- `HasFactory`
- `Notifiable`
- `SoftDeletes`

### Controllers Structure
```
app/Http/Controllers/Api/
├── Student/
│   └── AuthController.php
├── Parent/
│   └── AuthController.php
└── Admin/
    └── AuthController.php
```

### Middleware
- `student.auth` → Validates student tokens
- `parent.auth` → Validates parent tokens
- `admin.auth` → Validates admin/user tokens

---

## API Endpoints

Base URL: `http://your-domain.com/api`

### Student Routes

#### Public Endpoints
```
POST /student/register
POST /student/login
```

#### Protected Endpoints (Requires Bearer Token)
```
GET    /student/profile
PUT    /student/profile
POST   /student/change-password
POST   /student/logout
POST   /student/logout-all
```

### Parent Routes

#### Public Endpoints
```
POST /parent/register
POST /parent/login
```

#### Protected Endpoints (Requires Bearer Token)
```
GET    /parent/profile
PUT    /parent/profile
POST   /parent/change-password
POST   /parent/logout
POST   /parent/logout-all
```

### Admin/User Routes

#### Public Endpoints
```
POST /admin/register
POST /admin/login
```

#### Protected Endpoints (Requires Bearer Token)
```
GET    /admin/profile
PUT    /admin/profile
POST   /admin/change-password
POST   /admin/logout
POST   /admin/logout-all
```

---

## API Request/Response Examples

### 1. Student Registration

**Request:**
```http
POST /api/student/register
Content-Type: application/json

{
    "admission_number": "STU001",
    "first_name": "John",
    "last_name": "Doe",
    "email": "john.doe@student.com",
    "password": "password123",
    "password_confirmation": "password123",
    "phone": "1234567890",
    "date_of_birth": "2005-01-15",
    "gender": "male",
    "class_id": 1,
    "section_id": 1
}
```

**Response:**
```json
{
    "success": true,
    "message": "Student registered successfully",
    "data": {
        "student": {
            "id": 1,
            "admission_number": "STU001",
            "first_name": "John",
            "last_name": "Doe",
            "email": "john.doe@student.com",
            "created_at": "2024-01-23T10:00:00.000000Z"
        },
        "token": "1|abc123xyz...",
        "token_type": "Bearer"
    }
}
```

### 2. Student Login

**Request:**
```http
POST /api/student/login
Content-Type: application/json

{
    "email": "john.doe@student.com",
    "password": "password123"
}
```

**Response:**
```json
{
    "success": true,
    "message": "Login successful",
    "data": {
        "student": {
            "id": 1,
            "admission_number": "STU001",
            "first_name": "John",
            "last_name": "Doe",
            "email": "john.doe@student.com",
            "class": {...},
            "section": {...}
        },
        "token": "2|xyz456abc...",
        "token_type": "Bearer"
    }
}
```

### 3. Get Student Profile (Protected)

**Request:**
```http
GET /api/student/profile
Authorization: Bearer 2|xyz456abc...
```

**Response:**
```json
{
    "success": true,
    "data": {
        "id": 1,
        "admission_number": "STU001",
        "first_name": "John",
        "last_name": "Doe",
        "email": "john.doe@student.com",
        "class": {...},
        "section": {...},
        "school": {...}
    }
}
```

### 4. Parent Registration

**Request:**
```http
POST /api/parent/register
Content-Type: application/json

{
    "first_name": "Jane",
    "last_name": "Doe",
    "email": "jane.doe@parent.com",
    "password": "password123",
    "password_confirmation": "password123",
    "phone": "9876543210",
    "occupation": "Engineer"
}
```

### 5. Admin/User Login

**Request:**
```http
POST /api/admin/login
Content-Type: application/json

{
    "email": "admin@school.com",
    "password": "admin123"
}
```

**Response:**
```json
{
    "success": true,
    "message": "Login successful",
    "data": {
        "user": {
            "id": 1,
            "name": "Admin User",
            "email": "admin@school.com",
            "role_type": "admin"
        },
        "token": "3|admin789...",
        "token_type": "Bearer",
        "role": "admin"
    }
}
```

---

## Token Abilities

Each user type receives tokens with specific abilities:

- **Students**: `['student']`
- **Parents**: `['parent']`
- **Admin/Users**: `['admin']`, `['teacher']`, `['staff']`, or `['principal']`

### Middleware Protection

Routes are protected by checking both:
1. Valid Sanctum token
2. Correct token ability for the user type

Example:
```php
Route::middleware(['auth:sanctum', 'student.auth'])->group(function () {
    // Only students can access these routes
});
```

---

## Security Features

1. **Password Hashing**: Automatic via `password` cast
2. **Token Revocation**: Logout deletes current/all tokens
3. **Separate Guards**: Each user type isolated
4. **Ability-Based Access**: Token abilities prevent cross-type access
5. **Account Status**: Inactive accounts blocked from login

---

## Testing with Postman/cURL

### Student Login Test
```bash
curl -X POST http://localhost/api/student/login \
  -H "Content-Type: application/json" \
  -d '{
    "email": "student@example.com",
    "password": "password123"
  }'
```

### Access Protected Route
```bash
curl -X GET http://localhost/api/student/profile \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### Change Password
```bash
curl -X POST http://localhost/api/student/change-password \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "current_password": "password123",
    "new_password": "newpassword123",
    "new_password_confirmation": "newpassword123"
  }'
```

---

## Database Setup

### Required Migrations

1. **Students Table** - Add `password` column:
```php
$table->string('password')->nullable();
```

2. **Parents Table** - Add `password` column:
```php
$table->string('password')->nullable();
```

3. **Users Table** - Standard Laravel users table with:
```php
$table->string('name');
$table->string('email')->unique();
$table->string('password');
$table->string('role_type'); // admin, teacher, staff, principal
$table->string('phone')->nullable();
$table->boolean('is_active')->default(true);
$table->rememberToken();
$table->timestamps();
```

4. **Personal Access Tokens** (Sanctum):
```bash
php artisan migrate
```

---

## Adding New Protected Routes

### For Students:
```php
Route::prefix('student')
    ->middleware(['auth:sanctum', 'student.auth'])
    ->group(function () {
        Route::get('/attendance', [StudentAttendanceController::class, 'index']);
        Route::get('/marks', [StudentMarksController::class, 'index']);
        Route::get('/timetable', [StudentTimetableController::class, 'show']);
    });
```

### For Parents:
```php
Route::prefix('parent')
    ->middleware(['auth:sanctum', 'parent.auth'])
    ->group(function () {
        Route::get('/children', [ParentChildrenController::class, 'index']);
        Route::get('/children/{id}/attendance', [ParentAttendanceController::class, 'show']);
    });
```

### For Admin:
```php
Route::prefix('admin')
    ->middleware(['auth:sanctum', 'admin.auth'])
    ->group(function () {
        Route::apiResource('students', StudentController::class);
        Route::apiResource('teachers', TeacherController::class);
    });
```

---

## Error Responses

### Validation Error
```json
{
    "message": "The email field is required.",
    "errors": {
        "email": ["The email field is required."]
    }
}
```

### Authentication Error
```json
{
    "message": "Unauthenticated."
}
```

### Authorization Error (Wrong User Type)
```json
{
    "success": false,
    "message": "Unauthorized. Student access required."
}
```

### Inactive Account
```json
{
    "success": false,
    "message": "Your account is inactive. Please contact admin."
}
```

---

## Configuration Files

### Sanctum Config (`config/sanctum.php`)
```php
'expiration' => null, // Tokens never expire (or set minutes)
'guard' => ['web'],
```

### CORS Setup
Add to `.env`:
```
SANCTUM_STATEFUL_DOMAINS=localhost,127.0.0.1
```

---

## Next Steps

1. **Run Migrations**: Add password columns to students/parents tables
2. **Test Endpoints**: Use Postman collection
3. **Implement Controllers**: Create resource controllers for each user type
4. **Add Permissions**: Implement role-based permissions
5. **API Documentation**: Generate OpenAPI/Swagger docs

---

## Support

For issues or questions:
- Check logs: `storage/logs/laravel.log`
- Test routes: `php artisan route:list`
- Clear cache: `php artisan optimize:clear`
