# JWT Multi-Authentication API Documentation

## Overview
This School ERP API implements **JWT (JSON Web Token)** authentication with three separate authentication systems:
- **Admin/Staff/Teachers** (Users table)
- **Students** (Students table)
- **Parents** (Parents table)

## Table of Contents
- [Installation & Configuration](#installation--configuration)
- [Authentication Endpoints](#authentication-endpoints)
- [Using JWT Tokens](#using-jwt-tokens)
- [API Examples](#api-examples)
- [Error Handling](#error-handling)
- [Token Refresh](#token-refresh)

---

## Installation & Configuration

### 1. JWT Package
Already installed: `tymon/jwt-auth`

### 2. Configuration Files
- **config/auth.php** - Guards and providers configured for admin, student, parent
- **config/jwt.php** - JWT settings (TTL, algo, etc.)
- **.env** - JWT_SECRET already generated

### 3. JWT Token Settings
Default TTL: **60 minutes**
Refresh TTL: **20160 minutes** (2 weeks)

To change token expiry, update in `.env`:
```env
JWT_TTL=60
JWT_REFRESH_TTL=20160
```

---

## Authentication Endpoints

### Base URL
```
http://localhost:8000/api
```

### Admin/Staff/Teachers Endpoints
| Method | Endpoint | Description | Auth Required |
|--------|----------|-------------|---------------|
| POST | `/admin/register` | Register new admin/staff | No |
| POST | `/admin/login` | Login admin/staff | No |
| GET | `/admin/profile` | Get profile | Yes |
| PUT | `/admin/profile` | Update profile | Yes |
| POST | `/admin/change-password` | Change password | Yes |
| POST | `/admin/refresh` | Refresh JWT token | Yes |
| POST | `/admin/logout` | Logout | Yes |

### Student Endpoints
| Method | Endpoint | Description | Auth Required |
|--------|----------|-------------|---------------|
| POST | `/student/register` | Register new student | No |
| POST | `/student/login` | Login student | No |
| GET | `/student/profile` | Get profile | Yes |
| PUT | `/student/profile` | Update profile | Yes |
| POST | `/student/change-password` | Change password | Yes |
| POST | `/student/refresh` | Refresh JWT token | Yes |
| POST | `/student/logout` | Logout | Yes |

### Parent Endpoints
| Method | Endpoint | Description | Auth Required |
|--------|----------|-------------|---------------|
| POST | `/parent/register` | Register new parent | No |
| POST | `/parent/login` | Login parent | No |
| GET | `/parent/profile` | Get profile | Yes |
| PUT | `/parent/profile` | Update profile | Yes |
| POST | `/parent/change-password` | Change password | Yes |
| POST | `/parent/refresh` | Refresh JWT token | Yes |
| POST | `/parent/logout` | Logout | Yes |

---

## Using JWT Tokens

### 1. Authentication Header
For all protected routes, include the JWT token in the Authorization header:

```http
Authorization: Bearer <your_jwt_token>
```

### 2. Token Response Structure
```json
{
  "success": true,
  "message": "Login successful",
  "data": {
    "user": {
      "id": 1,
      "name": "John Doe",
      "email": "john@example.com",
      "role_type": "admin"
    },
    "token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
    "token_type": "Bearer",
    "expires_in": 3600
  }
}
```

### 3. Token Custom Claims
Each user type has custom claims in the JWT:

**Admin:**
```json
{
  "type": "admin",
  "role_type": "admin|teacher|staff|principal",
  "email": "admin@example.com"
}
```

**Student:**
```json
{
  "type": "student",
  "admission_number": "2024001",
  "class_id": 5
}
```

**Parent:**
```json
{
  "type": "parent",
  "email": "parent@example.com"
}
```

---

## API Examples

### 1. Admin Registration
```bash
curl -X POST http://localhost:8000/api/admin/register \
  -H "Content-Type: application/json" \
  -d '{
    "name": "John Admin",
    "email": "admin@school.com",
    "password": "password123",
    "password_confirmation": "password123",
    "phone": "1234567890",
    "role_type": "admin"
  }'
```

**Response:**
```json
{
  "success": true,
  "message": "User registered successfully",
  "data": {
    "user": {
      "id": 1,
      "name": "John Admin",
      "email": "admin@school.com",
      "role_type": "admin"
    },
    "token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
    "token_type": "Bearer",
    "expires_in": 3600
  }
}
```

### 2. Student Login
```bash
curl -X POST http://localhost:8000/api/student/login \
  -H "Content-Type: application/json" \
  -d '{
    "email": "student@school.com",
    "password": "password123"
  }'
```

**Response:**
```json
{
  "success": true,
  "message": "Login successful",
  "data": {
    "student": {
      "id": 1,
      "admission_number": "2024001",
      "first_name": "Jane",
      "last_name": "Student",
      "email": "student@school.com",
      "class": {...},
      "section": {...}
    },
    "token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
    "token_type": "Bearer",
    "expires_in": 3600
  }
}
```

### 3. Get Profile (Protected Route)
```bash
curl -X GET http://localhost:8000/api/admin/profile \
  -H "Authorization: Bearer eyJ0eXAiOiJKV1QiLCJhbGc..."
```

**Response:**
```json
{
  "success": true,
  "data": {
    "id": 1,
    "name": "John Admin",
    "email": "admin@school.com",
    "role_type": "admin",
    "roles": [...]
  }
}
```

### 4. Refresh Token
```bash
curl -X POST http://localhost:8000/api/admin/refresh \
  -H "Authorization: Bearer eyJ0eXAiOiJKV1QiLCJhbGc..."
```

**Response:**
```json
{
  "success": true,
  "token": "eyJ0eXAiOiJKV1QiLCJhbGc... (new token)",
  "token_type": "Bearer",
  "expires_in": 3600
}
```

### 5. Change Password
```bash
curl -X POST http://localhost:8000/api/admin/change-password \
  -H "Authorization: Bearer eyJ0eXAiOiJKV1QiLCJhbGc..." \
  -H "Content-Type: application/json" \
  -d '{
    "current_password": "oldpassword123",
    "new_password": "newpassword123",
    "new_password_confirmation": "newpassword123"
  }'
```

### 6. Logout
```bash
curl -X POST http://localhost:8000/api/admin/logout \
  -H "Authorization: Bearer eyJ0eXAiOiJKV1QiLCJhbGc..."
```

**Response:**
```json
{
  "success": true,
  "message": "Logged out successfully"
}
```

---

## Error Handling

### Common Error Responses

#### 1. Validation Error (422)
```json
{
  "success": false,
  "errors": {
    "email": ["The email field is required."],
    "password": ["The password must be at least 8 characters."]
  }
}
```

#### 2. Invalid Credentials (401)
```json
{
  "success": false,
  "message": "Invalid credentials"
}
```

#### 3. Unauthorized (401)
```json
{
  "message": "Unauthenticated."
}
```

#### 4. Token Expired (401)
```json
{
  "message": "Token has expired"
}
```

#### 5. Invalid Token (401)
```json
{
  "message": "Token is invalid"
}
```

#### 6. Account Inactive (403)
```json
{
  "success": false,
  "message": "Your account is inactive. Please contact administrator."
}
```

---

## Token Refresh

### When to Refresh
- Token expires after 60 minutes (default)
- Refresh before expiration to maintain session
- Use the `/refresh` endpoint to get a new token

### Refresh Flow
```javascript
// JavaScript example
async function refreshToken() {
  try {
    const response = await fetch('http://localhost:8000/api/admin/refresh', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${oldToken}`,
        'Content-Type': 'application/json'
      }
    });

    const data = await response.json();

    if (data.success) {
      // Store new token
      localStorage.setItem('token', data.token);
      return data.token;
    }
  } catch (error) {
    // Handle error, redirect to login
    console.error('Token refresh failed:', error);
  }
}
```

### Auto-Refresh Strategy
Implement token refresh 5 minutes before expiration:
```javascript
// Check token expiration and refresh
setInterval(() => {
  const tokenExpiry = getTokenExpiry(); // Decode JWT to get exp
  const now = Date.now() / 1000;

  if (tokenExpiry - now < 300) { // 5 minutes before expiry
    refreshToken();
  }
}, 60000); // Check every minute
```

---

## Security Best Practices

### 1. HTTPS
Always use HTTPS in production to prevent token interception.

### 2. Token Storage
**Frontend:**
- Use `httpOnly` cookies (recommended)
- Or `localStorage` (easier but less secure)
- Never use `sessionStorage` for JWT

### 3. Token Expiration
- Keep default TTL short (60 minutes)
- Use refresh tokens for longer sessions
- Implement logout on all devices feature

### 4. CORS Configuration
Update `config/cors.php` for your frontend domains:
```php
'allowed_origins' => [
    'http://localhost:3000',
    'https://yourapp.com',
],
```

### 5. Rate Limiting
Apply rate limiting to auth endpoints to prevent brute force:
```php
// In routes/api.php
Route::middleware('throttle:5,1')->group(function () {
    Route::post('/admin/login', [AdminAuthController::class, 'login']);
    Route::post('/student/login', [StudentAuthController::class, 'login']);
    Route::post('/parent/login', [ParentAuthController::class, 'login']);
});
```

---

## Testing with Postman

### 1. Import Collection
Import the provided `POSTMAN_COLLECTION_JWT.json` file

### 2. Set Environment Variables
Create a Postman environment with:
```
base_url: http://localhost:8000/api
admin_token: (will be set after login)
student_token: (will be set after login)
parent_token: (will be set after login)
```

### 3. Auto-Set Token
In login requests, add this to "Tests" tab:
```javascript
if (pm.response.code === 200) {
    const response = pm.response.json();
    pm.environment.set("admin_token", response.data.token);
}
```

---

## Troubleshooting

### Issue 1: "Token has expired"
**Solution:** Use the `/refresh` endpoint to get a new token

### Issue 2: "The token could not be parsed from the request"
**Solution:** Ensure Authorization header format is: `Bearer <token>`

### Issue 3: "User not found"
**Solution:** Token is valid but user was deleted. Re-authenticate.

### Issue 4: "Token is blacklisted"
**Solution:** Token was logged out. Need to login again.

### Issue 5: 500 Error on login
**Solution:** Check JWT_SECRET is set in .env file

---

## Additional Resources

### JWT Documentation
- Official Docs: https://jwt-auth.readthedocs.io/
- JWT.io: https://jwt.io/ (decode tokens)

### Laravel Resources
- Laravel Auth: https://laravel.com/docs/authentication
- Laravel Guards: https://laravel.com/docs/authentication#adding-custom-guards

---

## Support

For issues or questions:
1. Check error logs: `storage/logs/laravel.log`
2. Run `php artisan config:clear` after config changes
3. Run `php artisan route:clear` after route changes

---

**Last Updated:** January 2026
**Version:** 1.0.0
