# Database Connection Issue - Solutions

## Current Error
```
SQLSTATE[HY000] [1045] Access denied for user 'pffmstjc_schoolErp'@'106.219.166.69' (using password: YES)
```

## What Was Fixed
✅ **Session Driver Changed**: Changed from `database` to `file` in `.env`
✅ **Cache Driver Changed**: Changed from `database` to `file` in `.env`
✅ **Queue Driver Changed**: Changed from `database` to `sync` in `.env`

**Result**: Your JWT API will now work without database access for sessions/cache. The database is only needed for actual data operations.

---

## Current Database Configuration

```env
DB_HOST=157.10.98.169
DB_PORT=3306
DB_DATABASE=pffmstjc_schoolErp
DB_USERNAME=pffmstjc_schoolErp
DB_PASSWORD=pffmstjc_schoolErp@123#
```

**Issue**: The MySQL server at `157.10.98.169` is rejecting connections from your IP `106.219.166.69`.

---

## Solutions

### Solution 1: Fix Database Server Permissions (Recommended)

You need to grant access to your IP on the MySQL server:

**Option A: Using cPanel/WHM**
1. Login to cPanel or WHM
2. Go to "Remote MySQL"
3. Add your IP address: `106.219.166.69`
4. Or add `%` to allow all IPs (less secure)

**Option B: Using MySQL Command**
```sql
-- Login to MySQL on server 157.10.98.169
mysql -u root -p

-- Grant access from your IP
GRANT ALL PRIVILEGES ON pffmstjc_schoolErp.*
TO 'pffmstjc_schoolErp'@'106.219.166.69'
IDENTIFIED BY 'pffmstjc_schoolErp@123#';

-- Or grant from any IP (less secure)
GRANT ALL PRIVILEGES ON pffmstjc_schoolErp.*
TO 'pffmstjc_schoolErp'@'%'
IDENTIFIED BY 'pffmstjc_schoolErp@123#';

FLUSH PRIVILEGES;
```

**Option C: Check Firewall**
Make sure port 3306 is open on the server firewall for your IP:
```bash
# On the database server (157.10.98.169)
sudo ufw allow from 106.219.166.69 to any port 3306
```

---

### Solution 2: Use Local Database (For Development)

If you're developing locally, use a local MySQL/MariaDB database:

**1. Update `.env`:**
```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=schoolErp
DB_USERNAME=root
DB_PASSWORD=your_local_password
```

**2. Create local database:**
```bash
mysql -u root -p
CREATE DATABASE schoolErp;
exit;
```

**3. Run migrations:**
```bash
php artisan migrate
```

---

### Solution 3: Use SQLite (For Quick Testing)

For quick testing without MySQL:

**1. Update `.env`:**
```env
DB_CONNECTION=sqlite
# Comment out MySQL settings
# DB_HOST=157.10.98.169
# DB_PORT=3306
# DB_DATABASE=pffmstjc_schoolErp
# DB_USERNAME=pffmstjc_schoolErp
# DB_PASSWORD=pffmstjc_schoolErp@123#
```

**2. Create SQLite database:**
```bash
touch database/database.sqlite
php artisan migrate
```

---

### Solution 4: SSH Tunnel (Temporary Access)

If you can SSH to the server, create a tunnel:

```bash
# Create SSH tunnel
ssh -L 3306:157.10.98.169:3306 user@157.10.98.169

# Update .env
DB_HOST=127.0.0.1
```

---

## Testing Your JWT API Without Database

Your **JWT authentication will work** even without database access initially, because:

1. **JWT is stateless** - tokens are self-contained
2. **Sessions are file-based** - no database needed
3. **Cache is file-based** - no database needed

However, you'll need database access for:
- User registration (storing users)
- User login (fetching users)
- All CRUD operations

---

## Quick Test Commands

### 1. Test if configuration is loaded
```bash
php artisan config:show database
```

### 2. Test database connection
```bash
php artisan tinker --execute="DB::connection()->getPdo(); echo 'Success!';"
```

### 3. Test API endpoints (no database needed for this)
```bash
curl http://localhost:8000/api
```

### 4. Clear all caches
```bash
php artisan config:clear
php artisan cache:clear
php artisan route:clear
php artisan view:clear
```

---

## Current Status

✅ **Session Error Fixed**: Changed from database to file sessions
✅ **JWT Configuration**: Complete and ready
✅ **API Routes**: All configured
✅ **Controllers**: All created with JWT

❌ **Database Connection**: Needs remote MySQL access permissions

---

## Recommended Action Plan

**For Development:**
1. Use local MySQL database (Solution 2)
2. Run migrations to create tables
3. Test API with local data

**For Production:**
1. Contact hosting provider or server admin
2. Add your IP to MySQL remote access whitelist
3. Verify firewall allows port 3306
4. Test connection with MySQL client

---

## Contact Database Administrator

If this is a shared hosting or managed server, contact your administrator with:

**Information Needed:**
- Server: `157.10.98.169`
- Database: `pffmstjc_schoolErp`
- User: `pffmstjc_schoolErp`
- Your IP: `106.219.166.69`
- Port: `3306`

**Request:**
"Please grant remote MySQL access for user `pffmstjc_schoolErp` from IP `106.219.166.69` to database `pffmstjc_schoolErp` on port 3306."

---

## Next Steps

Choose one of the solutions above based on your environment:

- **Development**: Use Solution 2 (Local Database)
- **Testing**: Use Solution 3 (SQLite)
- **Production**: Use Solution 1 (Fix Permissions)

After database connection is established:
```bash
# Run migrations
php artisan migrate

# Seed database (if you have seeders)
php artisan db:seed

# Test authentication
curl -X POST http://localhost:8000/api/admin/register \
  -H "Content-Type: application/json" \
  -d '{"name":"Admin","email":"admin@test.com","password":"password123","password_confirmation":"password123","role_type":"admin"}'
```
